<?php

/**
 * @file The global functions for the Barion API module.
 * @author Müller Péter <egyeves@gmail.com>
 */

/**
 * Check if a value is an integer.
 * @param mixed $variable The value.
 * @return boolean Returns TRUE if the value is an integer else returns FALSE.
 * @author Müller Péter <egyeves@gmail.com>
 */
function _propeditor_is_int($variable) {
    return ctype_digit(strval($variable));
}

/**
 * Change all accent letter to similar english letters.
 * @param mixed $field The value.
 * @return string.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _propeditor_change_letters($field) {
    $letters = [
        0 => "a à á â ä æ ã å ā",
        1 => "c ç ć č",
        2 => "e é è ê ë ę ė ē",
        3 => "i ī į í ì ï î",
        4 => "l ł",
        5 => "n ñ ń",
        6 => "o ō ø œ õ ó ò ö ô ő",
        7 => "s ß ś š",
        8 => "u ū ú ù ü û ű",
        9 => "w ŵ",
        10 => "y ŷ ÿ",
        11 => "z ź ž ż",
    ];
    foreach ($letters as &$values) {
        $newValue = substr($values, 0, 1);
        $values = substr($values, 2, strlen($values));
        $values = explode(" ", $values);
        foreach ($values as &$oldValue) {
            while (strpos($field, $oldValue) !== false) {
                $field = preg_replace("/" . $oldValue . '/', $newValue, $field, 1);
            }
        }
    }
    $field = str_replace(' ', '-', $field);
    return $field;
}

/**
 * Check the email if correct or not.
 * @param string $email The email raw input.
 * @return boolean TRUE if the email looks correct.
 * @author Müller Péter <egyeves@gmail.com>
 */
function propeditor_is_valid_email($email) {
    $isValid = true;
    $atIndex = strrpos($email, "@");
    if (is_bool($atIndex) && !$atIndex) {
        $isValid = false;
    } else {
        $domain = substr($email, $atIndex + 1);
        $local = substr($email, 0, $atIndex);
        $localLen = strlen($local);
        $domainLen = strlen($domain);
        if ($localLen < 1 || $localLen > 64) {
            // Local part length exceeded.
            $isValid = false;
        } elseif ($domainLen < 1 || $domainLen > 255) {
            // Domain part length exceeded.
            $isValid = false;
        } elseif ($local[0] == '.' || $local[$localLen - 1] == '.') {
            // Local part starts or ends with '.'.
            $isValid = false;
        } elseif (preg_match('/\\.\\./', $local)) {
            // Local part has two consecutive dots.
            $isValid = false;
        } elseif (!preg_match('/^[A-Za-z0-9\\-\\.]+$/', $domain)) {
            // Character not valid in domain part.
            $isValid = false;
        } elseif (preg_match('/\\.\\./', $domain)) {
            // Domain part has two consecutive dots.
            $isValid = false;
        } elseif (!preg_match('/^(\\\\.|[A-Za-z0-9!#%&`_=\\/$\'*+?^{}|~.-])+$/', str_replace("\\\\", "", $local))) {
            // Character not valid in local part unless local part is quoted.
            if (!preg_match('/^"(\\\\"|[^"])+"$/', str_replace("\\\\", "", $local))) {
                $isValid = false;
            }
        }
        if ($isValid && !(checkdnsrr($domain, "MX") || checkdnsrr($domain, "A"))) {
            // Domain not found in DNS.
            $isValid = false;
        }
    }
    return $isValid;
}

/**
 * Validates e-mail in the Drupal Form API fields.
 * @param type $element
 * @param type $form_state
 * @author Müller Péter <egyeves@gmail.com>
 */
//function propeditor_element_validate_email($element, &$form_state) {
//    $value = trim($element['#value']);
//    if (!propeditor_is_valid_email($value)) {
//        form_error($element, t('%name must be an email.', array('%name' => $element['#title'])));
//    }
//}

/**
 * Creates a random alphanumeric string the given length.
 * @param integer $length Lenght of the string.
 * @param string $type The hash type.
 * @return string The random string.
 * @author Müller Péter <egyeves@gmail.com>
 */
function propeditor_make_hash($length = 32, $type = 'alfanumeric') {
    mt_srand((double) microtime() * 1000000);
    if ($type == 'alfanumeric') {
        $aZ09 = array_merge(range('A', 'Z'), range('a', 'z'), range(0, 9));
    } elseif ($type == 'alfa') {
        $aZ09 = array_merge(range('A', 'Z'), range('a', 'z'));
    } elseif ($type == 'bigalfa') {
        $aZ09 = array_merge(range('A', 'Z'));
    } elseif ($type == 'smallalfa') {
        $aZ09 = array_merge(range('a', 'z'));
    } else { // 'numeric'.
        $aZ09 = array_merge(range(0, 9));
    }
    $hash = '';
    for ($c = 0; $c < $length; $c++) {
        $hash .= $aZ09[mt_rand(0, count($aZ09) - 1)];
    }
    return $hash;
}

/**
 * If the IP is real it returns.
 * @return string The IP address if the IP is real else empty string.
 * @author Müller Péter <egyeves@gmail.com>
 */
function propeditor_get_real_ip() {
    $ip = $_SERVER['REMOTE_ADDR'];
    $ip = ip2long($ip);
    if ($ip !== false && $ip !== -1 && $ip !== 0) {
        $ip = long2ip($ip);
    } else {
        $ip = '';
    }
    return $ip;
}

/**
 * Easy look debug function.
 * @params type $var array.
 * @params type $json boolean flag.
 * @return string Formatted array tree.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function pdd($var, $json = FALSE) {
    if ($json) {
        header('Content-Type: application/json');
        echo drupal_json_output($var);
        exit();
    } else {
        echo '<pre>';
        print_r($var);
        echo '</pre>';
    }
}

/**
 * If the variable is defined, set new value.
 * @param type $form_state It has to contain [values] associative array
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_variable_set_from_form($form_state) {
    foreach ($form_state['values'] as $key => $value) {
        if (variable_get($key, 'no_var') !== 'no-var') {
            $value = propeditor_array_filter_empty_value($value, $form_state);
            variable_set($key, $value);
        }
    }
}

/**
 * Filter the array if the value is '' or null or false give back empty string;
 * @param type $form_state It has to contain [values] associative array
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_array_filter_empty_value($value, $form_state) {
    if (is_array($value) && isset($form_state['values']['array_filter'])) {
        $value = array_keys(array_filter($value));
        return $value;
    }
    return $value;
}

/**
 * Created a hash code.
 * @return string A hash code with 2 char and 6 number.
 * @author Müller Péter <egyeves@gmail.com>
 */
function propeditor_get_hash() {
    return propeditor_make_hash($length = 2, $type = 'bigalfa') . propeditor_make_hash($length = 6, $type = 'numeric');
}

/**
 * Return the unique hash code.
 * @return string The unique has code.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function propeditor_get_component_unique_hash() {
    $hash = propeditor_get_hash();
    // Iterates until the hash code not unique.
    $unique = db_query_range('SELECT cid FROM {propeditor_components} WHERE hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    while (!$unique) {
        $hash = propeditor_get_hash();
        $unique = propeditor_is_component_hash_unique($hash);
    }
    return $hash;
}

/**
 * Check if the has code is unique.
 * @param string $hash The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function propeditor_is_component_hash_unique($hash) {
    $result = db_query_range('SELECT cid FROM {propeditor_components} WHERE hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    return (empty($result)) ? TRUE : FALSE;
}

/**
 * Return the unique hash code.
 * @return string The unique has code.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function propeditor_get_template_unique_hash() {
    $hash = propeditor_get_hash();
    // Iterates until the hash code not unique.
    $unique = db_query_range('SELECT tid FROM {propeditor_templates} WHERE hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    while (!$unique) {
        $hash = propeditor_get_hash();
        $unique = propeditor_is_template_hash_unique($hash);
    }
    return $hash;
}

/**
 * Check if the has code is unique.
 * @param string $hash The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function propeditor_is_template_hash_unique($hash) {
    $result = db_query_range('SELECT tid FROM {propeditor_templates} WHERE hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    return (empty($result)) ? TRUE : FALSE;
}

/**
 * Return the unique hash code.
 * @return string The unique has code.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function propeditor_get_proposals_unique_hash() {
    $hash = propeditor_get_hash();
    // Iterates until the hash code not unique.
    $unique = db_query_range('SELECT prid FROM {propeditor_proposals} WHERE hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    while (!$unique) {
        $hash = propeditor_get_hash();
        $unique = propeditor_is_proposal_hash_unique($hash);
    }
    return $hash;
}

/**
 * Check if the has code is unique.
 * @param string $hash The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function propeditor_is_proposal_hash_unique($hash) {
    $result = db_query_range('SELECT prid FROM {propeditor_proposals} WHERE hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    return (empty($result)) ? TRUE : FALSE;
}

/**
 * Return the unique hash code.
 * @return string The unique has code.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function propeditor_get_email_stat_unique_hash() {
    $hash = propeditor_make_hash(12);
    // Iterates until the hash code not unique.
    $unique = db_query_range('SELECT esid FROM {propeditor_stat_emails} WHERE hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    while (!$unique) {
        $hash = propeditor_make_hash(12);
        $unique = propeditor_is_email_stat_hash_unique($hash);
    }
    return $hash;
}

/**
 * Check if the has code is unique.
 * @param string $hash The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function propeditor_is_email_stat_hash_unique($hash) {
    $result = db_query_range('SELECT esid FROM {propeditor_stat_emails} WHERE hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    return (empty($result)) ? TRUE : FALSE;
}

/**
 * Price value formatter.
 * @param type number.
 * @return string Formatted.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_format_price($number) {
    return number_format($number, 0, ',', ' ') . ' Ft';
}

/**
 * Check if a string is serialized
 * @param string $string
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_is_serial($string) {
    return (@unserialize($string) !== false);
}

/**
 * Check if a string is valid token for templates
 * @param string $string
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_is_token($string) {
    return preg_match('/^[a-zA-Z0-9_]+$/', $string);
}

/**
 * Check if a string is serialized
 * @param string $bigger
 * @param string $smaller
 * @return absolute value number.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_abs_different($bigger, $smaller) {
    return abs(floatval($bigger) - floatval($smaller));
}

/**
 * Get the difference between to dates in days
 * @param string $bigger
 * @param string $smaller
 * @return absolute value number.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_date_difference_days($bigger, $smaller) {

    $ts1 = strtotime($bigger);
    $ts2 = strtotime($smaller);

    $seconds_diff = $ts2 - $ts1;

    $days = abs(floor($seconds_diff / 3600 / 24));

    return $days;
}

/**
 * Check if the uid is valid.
 * @param string $uid The hash code.
 * @return boolean FALSE if the uid is not valid.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_isset_uid($uid) {
    $result = db_query_range('SELECT count(uid) uid FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid))->fetchField();
    return (empty($result)) ? FALSE : TRUE;
}

/**
 * Clean post, get or any key value pair.
 * @params array $data
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_clean_array($data) {

    $post = [];

    foreach ($data as $key => $value) {
        if (is_array($value)) {
            $post[check_plain(trim($key))] = propeditor_clean_array($value);
        } else {
            $post[check_plain(trim($key))] = check_plain(trim($value));
        }
    }

    return $post;
}

/**
 * Get component.
 * @params number $cid
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_component($cid, $column = 'cid') {

    $component = db_select('propeditor_components', 'c')
            ->condition($column, $cid)
            ->fields('c')
            ->execute()
            ->fetchObject();

    return $component;
}

/**
 * Get template.
 * @params number $tid
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_template($tid, $column = 'tid') {

    $template = db_select('propeditor_templates', 't')
            ->condition($column, $tid)
            ->fields('t')
            ->execute()
            ->fetchObject();

    return $template;
}

/**
 * Get template.
 * @params number $tid
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_templates($uid) {

    $db_or = db_or();
    $db_or->condition('uid', $uid);
    $db_or->condition('uid', 0);

    $templates = db_select('propeditor_templates', 't')
            ->condition($db_or)
            ->fields('t')
            ->execute()
            ->fetchAll();

    return $templates;
}

/**
 * Get proposal.
 * @params number $tid
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_proposal($prid, $column = 'prid') {

    $query = db_select('propeditor_proposals', 'p')
            ->condition('p.' . $column, $prid)
            ->fields('p');
    $query->leftJoin('propeditor_templates', 't', '(p.tid = t.tid)');
    $query->fields('t', array('tid', 'content', 'thumbnail'));
    $proposal = $query->execute()->fetchObject();

    $db_or = db_or();
    $db_or->condition('uid', $proposal->uid);
    $db_or->condition('uid', 0);
    $token_list = db_select('propeditor_tokens', 't')
            ->condition($db_or)
            ->fields('t', array('token'))
            ->execute()
            ->fetchAllKeyed(0, 0);
    $token_keys = array_keys($token_list);

    foreach (unserialize($proposal->tokens) as $key => $value) {
        $token = '%' . strtoupper(_propeditor_change_letters(trim($key))) . '%';
        if (!in_array($key, $token_keys) && !in_array($token, $token_keys)) {
            $key = propeditor_token_merge($token, $proposal->uid)['token'];
        }
        $token_list[$key] = $value;
    }

    $proposal->tokens = $token_list;
    $proposal->emails = unserialize($proposal->emails);

    return $proposal;
}

/**
 * Get proposal to edit.
 * @params number $tid
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_proposal_edit($prid, $column = 'prid') {

    $query = db_select('propeditor_proposals', 'p')
            ->condition('p.' . $column, $prid)
            ->fields('p', array('prid', 'tid', 'hash', 'uid', 'title'));
    $query->leftJoin('propeditor_templates', 't', '(p.tid = t.tid)');
    $query->fields('t', array('content', 'hash'));
    $proposal = $query->execute()->fetchObject();

    return $proposal;
}

/**
 * Get all proposal.
 * @params number $tid
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_all_proposal($uid) {

    $query = db_select('propeditor_proposals', 'p')
            ->condition('p.uid', $uid)
            ->fields('p');
    $query->leftJoin('propeditor_templates', 't', '(p.tid = t.tid)');
    $query->fields('t', array('tid', 'title', 'content'));
    $proposals = $query->execute()->fetchAll();

    return $proposals;
}

/**
 * Get token.
 * @params string $token
 * @params number $uid
 * @params enum $mode 'plain' or 'full'
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_token($token, $uid = 0, $mode = 'plain') {

    if ($mode == 'plain') {
        $token = '%' . strtoupper(_propeditor_change_letters(trim($token))) . '%';
    }

    $db_or = db_or();
    $db_or->condition('uid', $uid);
    $db_or->condition('uid', 0);

    $result = db_select('propeditor_tokens', 't')
            ->condition($db_or)
            ->condition('token', $token)
            ->fields('t')
            ->execute()
            ->fetchObject();

    return $result;
}

/**
 * Get all token of user.
 * @params number $uid
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_token_list($uid = 0) {

    $db_or = db_or();
    $db_or->condition('uid', $uid);
    $db_or->condition('uid', 0);

    $result = db_select('propeditor_tokens', 't')
            ->condition($db_or)
            ->fields('t')
            ->execute()
            ->fetchAll();

    return $result;
}

/**
 * Get proposal tokens array
 * @params number $proposal
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_proposal_token_list($proposal) {

    $db_or = db_or();
    $db_or->condition('uid', $proposal->uid);
    $db_or->condition('uid', 0);
    $token_list = db_select('propeditor_tokens', 't')
            ->condition($db_or)
            ->fields('t', array('token'))
            ->execute()
            ->fetchAllKeyed(0, 0);
    $token_keys = array_keys($token_list);

    foreach (unserialize($proposal->tokens) as $key => $value) {
        $token = '%' . strtoupper(_propeditor_change_letters(trim($key))) . '%';
        if (!in_array($key, $token_keys) && !in_array($token, $token_keys)) {
            $key = propeditor_token_merge($token, $proposal->uid)['token'];
        }
        $token_list[$key] = $value;
    }

    return $token_list;
}

/**
 * Save token to user
 * @param string $token
 * @param number $uid
 * @return array
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_token_create($token, $uid = false) {

    global $user;
    // Debug
    $user = user_load(1);

    if (!$user->uid && $uid === false) {
        return [
            'request' => [],
            'result' => [
                'msg' => 'Missing authetication!'
            ],
            'success' => false
        ];
    } else if (empty($token)) {
        return [
            'request' => $token,
            'result' => [
                'msg' => 'Missing data!'
            ],
            'success' => false
        ];
    } else if (!propeditor_is_token($token)) {
        return [
            'request' => $token,
            'result' => [
                'check' => propeditor_is_token($token),
                'msg' => 'Invalid token format!'
            ],
            'success' => false
        ];
    }

    $token = '%' . strtoupper(_propeditor_change_letters(trim($token))) . '%';

    // Make fields
    $fields = [
        'uid' => $uid === false ? $user->uid : $uid,
        'token' => $token,
        'created_at' => date('Y-m-d H:i:s'),
    ];

    // Merge token
    db_merge('propeditor_tokens')
            ->key(['token' => $token, 'uid' => $fields['uid']])
            ->fields($fields)
            ->execute();

    $db_or = db_or();
    $db_or->condition('uid', $fields['uid']);
    $db_or->condition('uid', 0);

    $tokens = db_select('propeditor_tokens', 't')
            ->condition($db_or)
            ->fields('t')
            ->execute()
            ->fetchAll();

    return [
        'request' => $token,
        'result' => $tokens,
        'success' => true
    ];
}

/**
 * Save token to user
 * @param string $token
 * @param number $uid
 * @param enum $mode 'plain' or 'full
 * @return array
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_token_merge($token, $uid, $mode = 'plain') {

    if ($mode == 'plain') {
        $token = '%' . strtoupper(_propeditor_change_letters(trim($token))) . '%';
    }

    // Make fields
    $fields = [
        'uid' => $uid,
        'token' => $token,
        'created_at' => date('Y-m-d H:i:s'),
    ];

    // Merge token
    db_merge('propeditor_tokens')
            ->key(['token' => $token, 'uid' => $fields['uid']])
            ->fields($fields)
            ->execute();

    return $fields;
}

/**
 * Delete user token.
 * @param string $token
 * @param enum $mode 'plain' or 'full'
 * @param number $uid
 * @return array
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_token_destroy($token, $mode = 'plain', $uid = false) {

    global $user;
    // Debug
    $user = user_load(1);

    if (!$user->uid && $uid === false) {
        return [
            'request' => [],
            'result' => [
                'msg' => 'Missing authetication!'
            ],
            'success' => false
        ];
    } else if (empty($token)) {
        return [
            'request' => $token,
            'result' => [
                'msg' => 'Missing data!'
            ],
            'success' => false
        ];
    } else if (!propeditor_is_token($token)) {
        return [
            'request' => $token,
            'result' => [
                'msg' => 'Invalid token format!'
            ],
            'success' => false
        ];
    }

    if ($mode == 'plain') {
        $token = '%' . strtoupper(_propeditor_change_letters(trim($token))) . '%';
    }

    // Make fields
    $fields = [
        'uid' => $uid === false ? $user->uid : $uid,
        'token' => $token,
        'created_at' => date('Y-m-d H:i:s'),
    ];
    $token = propeditor_get_token($token, $fields['uid'], 'full');

    // Delete token
    if ($token->uid !== 0) {
        db_delete('propeditor_tokens')
                ->condition('toid', $token->toid)
                ->execute();
    }

    $db_or = db_or();
    $db_or->condition('uid', $fields['uid']);
    $db_or->condition('uid', 0);

    $tokens = db_select('propeditor_tokens', 't')
            ->condition($db_or)
            ->fields('t')
            ->execute()
            ->fetchAll();

    return [
        'request' => $token,
        'result' => $tokens,
        'success' => true
    ];
}

/**
 * Create jpeg file.
 * @param string $data Base64
 * @param string $dir File location
 * @param string $name File name
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_base64_to_jpeg($data, $dir, $name) {

    list($type, $data) = explode(';', $data);
    list(, $data) = explode(',', $data);
    $data = base64_decode($data);

    $filename = $dir . $name . '.jpeg';

    if (!file_put_contents($filename, $data)) {

        return FALSE;
    } else {

        return propeditor_realpath_to_url($filename, false);
    }
}

/**
 * Clean base64 string
 * @param string $input Base64
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_clean_base64($input) {

    // Explode base64 png
    list($type, $data) = explode(';', $input);
    list($encode, $data) = explode(',', $data);

    $data = base64_decode(check_plain($data));

    $output = $type . ';' . $encode . ',' . base64_encode($data);
    $output = str_replace(" ", "", $output);

    // Input validate
    if ($input != $output) {
        return false;
    }

    return $output;
}

/**
 * Get created file real path
 * @param string $path
 * @param boolean $relative
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_realpath_to_url($path, $relative = TRUE) {
    if ($relative) {
        return str_replace($_SERVER['DOCUMENT_ROOT'], '', $path);
    } else {
        return 'https://' . $_SERVER['HTTP_HOST'] . str_replace($_SERVER['DOCUMENT_ROOT'], '', $path);
    }
}

/**
 * Set proposal is_sent value to the specified value.
 * @param string $hash
 * @param integer $value
 * @param date $next_date
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_proposal_set_email_send($hash, $value, $next_date) {

    try {
        // Update proposal
        db_update('propeditor_proposals')
                ->condition('hash', $hash)
                ->fields(array('is_sent' => (int) $value, 'queued' => 0, 'next_send' => date('Y-m-d H:i:s', strtotime($next_date))))
                ->execute();

        return TRUE;
    } catch (exepction $e) {
        watchdog('propeditor', 'Ajánlat kiküldött emailek értékének frissítése sikertelen. Error: %options.', array('%options' => json_encode($e), WATCHDOG_WARNING));
        return FALSE;
    }
}

/**
 * Egy e-mailt küld a rendszerből a felhasználónak.
 *
 * @param string $name
 * A címzett neve.
 *
 * @param string $email
 * A címzett e-mail címe.
 *
 * @param string $subject
 * Az e-mail tárgya.
 *
 * @param string $body
 * Az e-mail üzetnet.
 *
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_html_email_send($name, $email, $subject, $body) {

    try {
        $settings_mail_from_name = variable_get('propeditor_sender_name');
        $settings_mail_from_email = variable_get('propeditor_sender_email');

        $to = '"' . '=?UTF-8?B?' . base64_encode($name) . '?=' . '" <' . $email . '>';

        //    $message = '<!DOCTYPE html PUBLIC "-//W3C//DTD XHTML 1.0 Transitional//EN" "http://www.w3.org/TR/xhtml1/DTD/xhtml1-transitional.dtd">
        //<html xmlns="http://www.w3.org/1999/xhtml">
        //    <head>6
        //        <meta http-equiv="Content-Type" content="text/html; charset=UTF-8" />
        //        <title>' . $subject . ' | ' . variable_get('site_name') . '</title>
        //        <meta name="viewport" content="width=device-width, initial-scale=1.0"/>
        //    </head>
        //    <body style="margin: 0; padding: 0;">';
        $message = '';
        $message .= $body;
        $message .= ''; // @TODO Opening pixel image.
        //    $message .= '   </body>
        //</html>';

        $header = 'MIME-Version: 1.0' . PHP_EOL;
        $header .= 'Content-type: text/html; charset="UTF-8"' . PHP_EOL;
        $header .= 'Content-Transfer-Encoding: 8bit' . PHP_EOL;
        $header .= 'From: "' . '=?UTF-8?B?' . base64_encode($settings_mail_from_name) . '?=' . '" <' . $settings_mail_from_email . '>' . PHP_EOL;
        $header .= 'Reply-To: ' . $settings_mail_from_email . PHP_EOL;
        $header .= 'Return-Path: ' . $settings_mail_from_email . PHP_EOL;
        $header .= 'X-Mailer: PHP-' . phpversion() . PHP_EOL;

        mail($to, '=?UTF-8?B?' . base64_encode($subject) . '?=', $message, $header);

        return TRUE;
    } catch (Exception $e) {
        watchdog('propeditor', 'Ajánlat email küldése sikertelen. Error: %options.', array('%options' => json_encode([$e, $name, $email, $subject, $body]), WATCHDOG_WARNING));
        return FALSE;
    }
}

/**
 * Replace tokens to values in a string
 * @param string $string
 * @param array $tokens
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function changeTokens($string, $tokens) {
    foreach ($tokens as $key => $value) {
        $string = str_replace($key, $value, $string);
    }
    return $string;
}

/**
 * Get email stat record.
 * @params string $esid
 * @params string $column
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_get_email_stat($esid, $column = 'esid') {

    $component = db_select('propeditor_stat_emails', 's')
            ->condition('s.' . $column, $esid)
            ->fields('s')
            ->execute()
            ->fetchObject();

    return $component;
}

/**
 * Get email stat record.
 * @params string $prid
 * @params string $eid
 * @params string $subject
 * @params string $content
 * @params string $reciever_name
 * @params string $reciever_email
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function propeditor_generate_email_stat($prid, $eid, $subject, $content, $reciever_name, $reciever_email) {

    try {
        $hash = propeditor_get_email_stat_unique_hash();

        $record = db_select('propeditor_stat_emails', 's')
                ->condition('s.prid', $prid)
                ->condition('s.eid', $eid)
                ->fields('s')
                ->execute()
                ->fetchObject();

        if (empty($record)) {

            $fields = [
                'hash' => $hash,
                'prid' => $prid,
                'eid' => $eid,
                'subject' => $subject,
                'content' => $content,
                'reciever_name' => $reciever_name,
                'reciever_email' => $reciever_email,
                'opens_log' => serialize([]),
                'opens' => 0,
                'clicks' => 0,
                'logins' => 0,
                'sent_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s'),
            ];

            // Insert component if it is created
            db_insert('propeditor_stat_emails')
                    ->fields($fields)
                    ->execute();
        } else {

            $fields = [
                'subject' => $subject,
                'content' => $content,
                'reciever_name' => $reciever_name,
                'reciever_email' => $reciever_email,
                'sent_at' => date('Y-m-d H:i:s'),
                'updated_at' => date('Y-m-d H:i:s'),
            ];

            // Insert component if it is created
            db_update('propeditor_stat_emails')
                    ->condition('esid', $record->esid)
                    ->fields($fields)
                    ->execute();

            $hash = $record->hash;
        }

        return $hash;
    } catch (Exeption $e) {
        watchdog('propeditor', 'Email statisztikai rekord létrehozása sikertelen. Error: %options.', array('%options' => json_encode($e), WATCHDOG_WARNING));
        return false;
    }
}
