<?php

/**
 * @file Ownership Module functions for Withdrawals.
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */

/**
 * Withdrawals listing.
 * @return type html table of Withdrawals.
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_withdrawal_list() {
    $html = '';

    $header = array(
        array('data' => 'wid', 'field' => 'wid', 'sort' => 'asc'),
        array('data' => t('Folyószámla neve'), 'field' => 'name'),
        array('data' => t('Összeg'), 'field' => 'percent'),
        array('data' => t('Fizetve'), 'field' => 'type'),
        array('data' => t('Módosítva'), 'field' => 'is_active'),
        array('data' => t('Műveletek')),
    );

    $query = db_select('ownership_withdrawals', 'ow');
    $query->join('ownership_accounts', 'oa', 'oa.aid = ow.aid');
    $query->condition('ow.deleted_at', NULL, 'IS NULL');
    $query->addField('ow', 'wid');
    $query->addField('oa', 'name');
    $query->addField('ow', 'amount');
    $query->addField('ow', 'payed_at');
    $query->addField('ow', 'updated_at');
    $result = $query
                    ->range(0, 100)
                    ->extend('TableSort')
                    ->orderByHeader($header)
                    ->execute()->fetchAll();

    $rows = array();

    foreach ($result as $value) {
        $links = l(t('Szerkesztés'), "admin/config/ownership/withdrawal/edit/$value->wid") .
                '&nbsp;&nbsp;' .
                l(t('Törlés'), "admin/config/ownership/withdrawal/delete/$value->wid") .
                '&nbsp;&nbsp;';

        $rows[] = array(
            'data' => array(
                $value->wid,
                $value->name,
                $value->amount,
                $value->payed_at,
                $value->updated_at,
                $links,
            )
        );
    }

    $html .= theme('table', array(
        'header' => $header,
        'rows' => $rows,
        'caption' => t('Lehívások listája'),
        'sticky' => TRUE,
        'empty' => t('Még nincsenek Lehívások létrehozva.'),
    ));

    $html .= theme('pager', array('tags' => array()));

    return $html;
}

/**
 * Defines the form main for Withdrawal.
 *
 * @return $form
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_withdrawal_base_form() {

    // Values for entity Account
    $select_accounts = db_select('ownership_accounts', 'oa');
    $select_accounts->fields('oa', array('aid', 'name'));
    $select_accounts->orderBy('name', 'ASC');

    $account_entries = $select_accounts->execute()->fetchAll();

    foreach ($account_entries as $a_entry) {
        $account_options[$a_entry->aid] = t($a_entry->name);
    }

    $form = array();

    $form['overview'] = array(
        '#prefix' => '<p>',
        '#suffix' => '</p>',
    );

    $form['account'] = array(
        '#title' => t('Folyószámla'),
        '#description' => t('A folyószámla neve'),
        '#type' => 'select',
        '#options' => $account_options,
        '#required' => TRUE,
    );

    $form['amount'] = array(
        '#title' => t('Összeg'),
        '#description' => t('A lehívás összege'),
        '#type' => 'textfield',
        '#maxlength' => 9,
        '#required' => TRUE,
    );

    $form['payed_at'] = array(
        '#title' => t('Kifizetés dátuma'),
        '#description' => t('A lehívás dátuma'),
        '#type' => 'date_popup',
        '#date_format' => 'Y-m-d H:i:s',
        '#required' => TRUE,
    );

    $form['submit'] = array(
        '#type' => 'submit',
        '#value' => t('Lehívás rögzítése'),
    );

    return $form;
}

/**
 * Implements hook_form().
 * Defines form for creating Withdrawal
 * @params $form id
 * @params &$form_state form values
 * @return $form
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_withdrawal_create_form($form, &$form_state) {
    $form = ownership_withdrawal_base_form();
    $form['overview']['#markup'] = t('Ezen a felületen lehet létrehozni új Lehívásokat.');
    return $form;
}

/**
 * Validates form submit for creating Withdrawal.
 * @params $form id
 * @params &$form_state form values
 * @return validated form values
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_withdrawal_create_form_validate($form, &$form_state) {
    $amount = $form_state['values']['amount'];

    if (!((preg_match('{^[0-9]{1,9}$}', $amount)) && ($amount >= 0))) {
        form_set_error('amount', t('Helytelen érték. Az összeg mezőben pozitív egész számot adj meg, ami nem hosszabb 9 számjegynél.'));
    }
}

/**
 * Submits form for creating Withdrawal.
 * @params $form id
 * @params &$form_state form values
 * @return redirect
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_withdrawal_create_form_submit($form, &$form_state) {
    global $user;
    date_default_timezone_set('Europe/Budapest');

    $withdrawal_fields = array(
        'uid' => $user->uid,
        'aid' => (int) trim($form_state['values']['account']),
        'amount' => (int) trim($form_state['values']['amount']),
        'payed_at' => $form_state['values']['payed_at'],
        'created_at ' => date("Y-m-d h:i:s"),
        'updated_at ' => date("Y-m-d h:i:s"),
    );

    $withdrawal_id = db_insert('ownership_withdrawals')->fields($withdrawal_fields)->execute();

    $form_state['redirect'] = array('admin/config/ownership/withdrawal',
        array(
            'query' => array(),
            'fragment' => '',
        ),
    );

    drupal_set_message(t('<em>@amount</em> összegű lehívás létrehozva.', array('@amount' => $withdrawal_fields['amount'])));

    // Save Watchdog notice.
    watchdog('ownership', '@amount összegű lehívás létrehozva.', array('@amount' => $withdrawal_fields['amount'], WATCHDOG_NOTICE));
}

/**
 * Implements hook_form().
 * Defines form for editing Withdrawal
 * @params $form id
 * @params &$form_state form values
 * @params $aid withdrawal id
 * @return $form
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_withdrawal_edit_form($form, &$form_state, $wid) {
    $withdrawal = db_select('ownership_withdrawals', 'ow')->fields('ow')->condition('wid', $wid)->execute()->fetchObject();

    $form = ownership_withdrawal_base_form();
    $form['overview']['#markup'] = t('Ezen a felületen lehet szerkeszteni a meglévő lehívásokat.');
    $form['account']['#default_value'] = $withdrawal->aid;
    $form['amount']['#default_value'] = $withdrawal->amount;
    $form['payed_at']['#default_value'] = $withdrawal->payed_at;

    return $form;
}

/**
 * Validates form submit for editing Withdrawal.
 * @params $form id
 * @params &$form_state form values
 * @return validated form values
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_withdrawal_edit_form_validate($form, &$form_state) {
    $amount = $form_state['values']['amount'];

    if (!((preg_match('{^[0-9]{1,9}$}', $amount)) && ($amount >= 0))) {
        form_set_error('amount', t('Helytelen érték. Az összeg mezőben pozitív egész számot adj meg, ami nem hosszabb 9 számjegynél.'));
    }
}

/**
 * Submits form for edited Withdrawal.
 * @params $form id
 * @params &$form_state form values
 * @return redirect
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_withdrawal_edit_form_submit($form, &$form_state) {
    global $user;
    date_default_timezone_set('Europe/Budapest');
    $wid = $form_state['build_info']['args'][0];

    $withdrawal_fields = array(
        'uid' => $user->uid,
        'amount' => (int) trim($form_state['values']['amount']),
        'payed_at' => $form_state['values']['payed_at'],
        'updated_at' => date("Y-m-d h:i:s"),
    );

    $withdrawal_updated = db_update('ownership_withdrawals')->fields($withdrawal_fields)->condition('wid', $wid, '=')->execute();

    //Redirect to the listing site.
    $form_state['redirect'] = array('admin/config/ownership/withdrawal',
        array(
            'query' => array(),
            'fragment' => '',
        ),
    );

    // Message for the User.
    drupal_set_message(t('<em>@amount</em> összegű lehívás adatai sikeresen módosításra kerültek.', array('@amount' => $withdrawal_fields['amount'])));


    // Save to Watchdog notice.
    watchdog('ownership', '@amount összegű lehívás adatai sikeresen megváltoztak.', array('@amount' => $withdrawal_fields['amount'], WATCHDOG_NOTICE));
}

/**
 * Implements hook_form().
 * Creates form for confirm deleting Withdrawals.
 * @params $form id
 * @params &$form_state form values
 * @params $wid withdrawal id
 * @return redirect
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_withdrawal_delete_form($form, &$form_state, $wid) {
    $result = db_select('ownership_withdrawals', 'ow')->fields('ow', array('amount'))->condition('ow.wid', $wid)->range(0, 1)->execute()->fetchField();

    $form = array();
    $form['wid'] = array(
        '#type' => 'value',
        '#value' => $wid,
    );

    // Keep the old Withdrawal amount for the Watchdog notice
    $form['amount'] = array(
        '#type' => 'hidden',
        '#value' => $result,
    );

    return confirm_form(
            $form, t('<em>"' . $result . '"</em> összegű lehívás biztosan törölhető?'), 'admin/config/ownership/withdrawal', t('Ez a művelet nem vonható vissza.'), t('Lehvás törlése'), t('Mégsem')
    );
}

/**
 * Submit for deleting Withdrawal.
 * @params $form id
 * @params &$form_state form values
 * @params $wid withdrawal id
 * @return redirect
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_withdrawal_delete_form_submit($form, &$form_state) {
    date_default_timezone_set('Europe/Budapest');
    $wid = array('name' => $form_state['values']['wid']);
    $amount = $form_state['values']['amount'];

    $withdrawal_fields = array(
        'deleted_at' => date("Y-m-d h:i:s"),
    );

    $withdrawal_deleted = db_update('ownership_withdrawals')->fields($withdrawal_fields)->condition('wid', $wid, '=')->execute();

    // Redirect to the listing page
    $form_state['redirect'] = array('admin/config/ownership/withdrawal',
        array(
            'query' => array(),
            'fragment' => '',
        ),
    );

    // Message for the User.
    drupal_set_message(t('Lehívás sikeresen törölve.'));

    // Save Watchdog notice.
    watchdog('ownership', '@amount összegű lehívás sikeresen törölve.', array('@amount' => $amount), WATCHDOG_NOTICE);
}
