<?php

/**
 * @file The global functions for the Certify module.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */

/**
 * Check if a value is an integer.
 * @param mixed $variable The value.
 * @return boolean Returns TRUE if the value is an integer else returns FALSE.
 * @author Müller Péter <egyeves@gmail.com>
 */
function _ownership_is_int($variable) {
    return ctype_digit(strval($variable));
}

/**
 * Get the fields form the certify_candidate_groups table.
 * @return string The name field value.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_get_user_fields($uid) {
    return db_query_range('SELECT uid, name, mail FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid,))->fetchAssoc();
}

/**
 * Easy look debug function.
 * @params type $var array.
 * @params type $json boolean flag.
 * @return string Formatted array tree.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function odd($var, $json = FALSE) {
    if ($json) {
        header('Content-Type: application/json');
        echo drupal_json_output($var);
        exit();
    } else {
        echo '<pre>';
        print_r($var);
        echo '</pre>';
    }
}

/**
 * Filter the array if the value is '' or null or false give back empty string;
 * @param type $form_state It has to contain [values] associative array
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_array_filter_empty_value($value, $form_state) {
    if (is_array($value) && isset($form_state['values']['array_filter'])) {
        $value = array_keys(array_filter($value));
        return $value;
    }
    return $value;
}

/**
 * Price value formatter.
 * @param type number.
 * @return string Formatted.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_format_price($number) {
    return number_format($number, 0, ',', ' ') . ' ' . variable_get('ownership_firm_invoice_currency', 'Ft');
}

/**
 * Check if a string is serialized
 * @param string $string
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_is_serial($string) {
    return (@unserialize($string) !== false);
}

/**
 * Check if a string is serialized
 * @param string $bigger
 * @param string $smaller
 * @return absolute value number.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_get_abs_different($bigger, $smaller) {
    return abs(floatval($bigger) - floatval($smaller));
}

/**
 * Check if the uid is valid.
 * @param string $uid The hash code.
 * @return boolean FALSE if the uid is not valid.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_isset_uid($uid) {
    $result = db_query_range('SELECT count(uid) uid FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid))->fetchField();
    return (empty($result)) ? FALSE : TRUE;
}

/**
 * Adds a new role
 * @machine_name - the name of the role to be added
 * @weight - the optional display weight for role
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_add_role($machine_name, $weight = 0) {
    $role = new stdClass();
    $role->name = $machine_name;
    $role->weight = $weight;
    user_role_save($role);
}

/**
 * Adds permissions to a role
 * @role_name - the name of the role to receive an update
 * @permissions - the array of permissions to add
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_add_permissions($role_name, $permissions) {
    $role = user_role_load_by_name($role_name);
    user_role_grant_permissions($role->rid, $permissions);
}

/**
 * Return the two digits string
 * @params type integer.
 * @return string.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_two_digits($int) {
    if (strlen($int) == 1) {
        return '0' . $int;
    } else {
        return $int;
    }
}

/**
 * Add time value to the given date.
 * @params type string date.
 * @params type string.
 * @params type integer.
 * @return timestamp.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_add_time($date, $type = 'days', $int = 1) {

    $new_date = strtotime("+" . $int . " " . $type, strtotime($date));
    return $new_date;
}

/**
 * Show full name in set order.
 * @params type string $lastname.
 * @params type string $firstname.
 * @return string Fullname.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _ownership_show_name($lastname, $firstname) {
    if (variable_get('ownership_basic_name', '1') == 1) {
        return _capitalize($firstname) . ' ' . _capitalize($lastname);
    } else {
        return _capitalize($lastname) . ' ' . _capitalize($firstname);
    }
}

/**
 * Get all active orders amount.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_orders_sum() {

    $query = db_select('membership_orders', 'o')
            ->condition('o.is_activated', null, 'IS NOT')
            ->fields('o', ['vat'])
            ->groupBy('o.vat');
    $query->addExpression('sum(o.price)', 'total');
    $result = $query->execute()->fetchObject();

    if ($result == false) {
        $result = new \stdClass();
        $result->total = 0;
        $result->vat = 0;
    }

    return $result;
}

/**
 * Get all active orders amount.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_orders_sum_per_vat() {

    $query = db_select('membership_orders', 'o')
            ->condition('o.is_activated', null, 'IS NOT')
            ->fields('o', ['vat'])
            ->groupBy('o.vat');
    $query->addExpression('sum(o.price)', 'total');
    $result = $query->execute()->fetchAll();

    if ($result == false) {
        $result = new \stdClass();
        $result->total = 0;
        $result->vat = 0;
        $result = [$result];
    }

    return $result;
}

/**
 * Get all active orders amount.
 * @param array Products array from account product filter.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_orders_and_periods_sum_by_pids_per_vat($products = []) {
    
    $orders = [];
    $periods = [];
    $product_pids = [];
    $period_pids = [];
    foreach($products as $k => $v) {
        $key = explode('_', $k);
        if ($v !== 0 && $key[0] === 'pp') {
            $period_pids[] = $key[1];
        } else if ($v !== 0) {
            $product_pids[] = $key[1];
        }
    }

    if ($product_pids !== []) {
        $query = db_select('membership_orders', 'o')
                ->condition('o.is_activated', null, 'IS NOT')
                ->condition('o.pid', $product_pids, 'IN')
                ->fields('o', ['vat'])
                ->groupBy('o.vat');
        $query->addExpression('sum(o.price)', 'total');
        $orders = $query->execute()->fetchAll();

        if ($orders == false) {
            $result = new \stdClass();
            $result->total = 0;
            $result->vat = 0;
            $orders = [$result];
        }
    }
    
    if (db_table_exists('membership_period_orders')) {
        if ($period_pids !== []) {
            $query = db_select('membership_period_orders', 'o')
                    ->condition('o.is_activated', null, 'IS NOT')
                    ->condition('o.pid', $period_pids, 'IN')
                    ->fields('o', ['vat'])
                    ->groupBy('o.vat');
            $query->addExpression('sum(o.price)', 'total');
            $periods = $query->execute()->fetchAll();

            if ($periods == false) {
                $result = new \stdClass();
                $result->total = 0;
                $result->vat = 0;
                $periods = [$result];
            }
        }
    } else {
        $result = new \stdClass();
        $result->total = 0;
        $result->vat = 0;
        $periods = [$result];
    }
    $output = [];
    if ($product_pids == []) {
        $output = $periods;
    } else if ($period_pids == []) {
        $output = $orders;
    } else {
        array_splice($periods, 0, 0, $orders);
        $output = $periods;
    }
    
    return $output;
}

/**
 * Get all active period orders amount.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_period_orders_sum() {
    
    if (db_table_exists('membership_period_orders')) {
        $query = db_select('membership_period_orders', 'o')
                ->condition('o.is_activated', null, 'IS NOT')
                ->fields('o', ['vat'])
                ->groupBy('o.vat');
        $query->addExpression('sum(o.price)', 'total');
        $result = $query->execute()->fetchObject();
    } else {
        $result = false;
    }

    if ($result == false) {
        $result = new \stdClass();
        $result->total = 0;
        $result->vat = 0;
    }

    return $result;
}

/**
 * Get all active period orders amount.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_period_orders_sum_per_vat() {

    if (db_table_exists('membership_period_orders')) {
        $query = db_select('membership_period_orders', 'o')
                ->condition('o.is_activated', null, 'IS NOT')
                ->fields('o', ['vat'])
                ->groupBy('o.vat');
        $query->addExpression('sum(o.price)', 'total');
        $result = $query->execute()->fetchAll();

        if ($result == false) {
            $result = new \stdClass();
            $result->total = 0;
            $result->vat = 0;
            $result = [$result];
        }
    } else {
        $result = new \stdClass();
        $result->total = 0;
        $result->vat = 0;
        $result = [$result];
    }

    return $result;
}

/**
 * Get all accounts total cost.
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_costs_sum() {

    $query = db_select('ownership_costs', 'c')
            ->condition('c.payed_at', null, 'IS NOT')
            ->fields('c', ['aid'])
            ->groupBy('c.aid');
    $query->addExpression('sum(c.amount)', 'total');
    $result = $query->execute()->fetchAllKeyed();

    return $result;
}

/**
 * Get all accounts total withdrawals.
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_withdrawals_sum() {

    $query = db_select('ownership_withdrawals', 'w')
            ->condition('w.payed_at', null, 'IS NOT')
            ->fields('w', ['wid'])
            ->groupBy('w.wid');
    $query->addExpression('sum(w.amount)', 'total');
    $result = $query->execute()->fetchAllKeyed();

    return $result;
}

/**
 * Get all account list.
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_account_list() {

    $query = db_select('ownership_accounts', 'a')
            ->fields('a')
            ->orderBy('a.priority', 'desc');
    $result = $query->execute()->fetchAll();

    return $result;
}

/**
 * Get all balance list.
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_balance_list() {

    $query = db_select('ownership_balances', 'b')
            ->fields('b')
            ->orderBy('updated_at', 'desc');
    $result = $query->execute()->fetchAll();

    return $result;
}

/**
 * Get balance.
 * @param integer $bid
 * @param string $column
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_balance($bid, $column = 'bid') {

    $query = db_select('ownership_balances', 'b')
            ->condition('b.' . $column, $bid)
            ->fields('b');
    $result = $query->execute()->fetchObject();

    return $result;
}

/**
 * Create balance
 * @param array $fields
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_create_balance($fields) {

    $defaults = array(
        'aid' => isset($fields['aid']) ? $fields['aid'] : 0,
        'open_amount' => isset($fields['open_amount']) ? $fields['open_amount'] : 0,
        'amount' => isset($fields['amount']) ? $fields['amount'] : 0,
        'created_at' => isset($fields['created_at']) ? $fields['created_at'] : date('Y-m-d H:i:s'),
        'updated_at' => isset($fields['updated_at']) ? $fields['updated_at'] : date('Y-m-d H:i:s'),
        'closed_at' => isset($fields['closed_at']) ? $fields['closed_at'] : null,
    );

    $bid = db_insert('ownership_balances')->fields($defaults)->execute();

    return $bid;
}

/**
 * Get current balance of account
 * @param integer $aid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_current_balance($aid) {

    $query = db_select('ownership_balances', 'b')
            ->condition('b.aid', $aid)
            ->orderBy('b.created_at', 'desc')
            ->fields('b');
    $balances = $query->execute()->fetchAll();

    if (!empty($balances) && $balances[0]->closed_at === null) { // Ha van egyenleg és nem lezárt
        // Visszaadjuk az utolsó egyenleget
        $result = $balances[0];
    } else if (!empty($balances) && $balances[0]->closed_at !== null) { // Ha van egyenleg, de lezárt
        // Készítünk aktuális egyenleget
        $fields = array(
            'aid' => $aid,
            'open_amount' => $balances[0]->amount,
            'amount' => $balances[0]->amount,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
        );
        $bid = ownership_create_balance($fields);
        $result = ownership_get_balance($bid);
    } else { // Ha nincs egyenleg
        // Készítünk aktuális egyenleget
        $fields = array(
            'aid' => $aid,
            'open_amount' => 0,
            'amount' => 0,
            'created_at' => date('Y-m-d H:i:s'),
            'updated_at' => date('Y-m-d H:i:s'),
        );
        $bid = ownership_create_balance($fields);
        $result = ownership_get_balance($bid);
    }

    return $result;
}

/**
 * Close balance
 * @param array $fields
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_close_balance($bid) {

    $balance = ownership_get_balance($bid);
    $fields = array(
        'closed_at' => date('Y-m-d H:i:s'),
    );

    db_update('ownership_balances')
            ->fields($fields)
            ->condition('bid', $bid)
            ->execute();

    $fields = array(
        'aid' => $balance->aid,
        'open_amount' => $balance->amount,
        'amount' => $balance->amount,
        'created_at' => date('Y-m-d H:i:s'),
        'updated_at' => date('Y-m-d H:i:s'),
    );
    $new_bid = ownership_create_balance($fields);

    return $new_bid;
}

/**
 * Get all membeship product
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_membership_products() {

    if (db_table_exists('membership_period_orders')) {
        $periods = db_select('membership_period_products', 'p')
                ->fields('p', array('pid', 'name', 'price', 'hash'))
                ->execute()
                ->fetchAll();
    } else {
        $periods = null;
    }
    
    $products = db_select('membership_products', 'p')
            ->fields('p', array('pid', 'name', 'price', 'hash'))
            ->execute()
            ->fetchAll();
    
    $output = compact('periods', 'products');

    return $output;
}

/**
 * Get all membeship product
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ownership_get_membership_products_list() {

    if (db_table_exists('membership_period_orders')) {
        $periods = db_select('membership_period_products', 'p')
                ->fields('p', array('pid', 'name', 'price', 'hash'))
                ->execute()
                ->fetchAll();
    } else {
        $periods = [];
    }
    
    $products = db_select('membership_products', 'p')
            ->fields('p', array('pid', 'name', 'price', 'hash'))
            ->execute()
            ->fetchAll();
    $output = [];
    
    foreach($products as $product) {
        $output['p_' . $product->pid] = $product->name . ' (' . _ownership_format_price($product->price) . ' - ' . $product->hash . ')';
    }
    
    foreach($periods as $product) {
        $output['pp_' . $product->pid] = $product->name . ' (' . _ownership_format_price($product->price) . ' - ' . $product->hash . ')';
    }

    return $output;
}