<?php

/**
 * @file Ownership Module functions for Costs.
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */

/**
 * Cost listing.
 * @return type html table of Costs.
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_cost_list() {
    $html = '';

    $header = array(
        array('data' => 'cid', 'field' => 'cid', 'sort' => 'asc'),
        array('data' => t('Összeg'), 'field' => 'amount'),
        array('data' => t('Folyószámla neve'), 'field' => 'name'),
        array('data' => t('Cím'), 'field' => 'title'),
        array('data' => t('Megjegyzés'), 'field' => 'description'),
        array('data' => t('Kifizetés'), 'field' => 'payed_at'),
    );

    $query = db_select('ownership_costs', 'oc');
    $query->join('ownership_accounts', 'oa', 'oa.aid = oc.aid');
    $query->addField('oc', 'cid');
    $query->addField('oc', 'amount');
    $query->addField('oa', 'name');
    $query->addField('oc', 'title');
    $query->addField('oc', 'description');
    $query->addField('oc', 'payed_at');
    $result = $query
                    ->range(0, 100)
                    ->extend('TableSort')
                    ->orderByHeader($header)
                    ->execute()->fetchAll();

    $rows = array();

    foreach ($result as $value) {

        $rows[] = array(
            'data' => array(
                $value->cid,
                $value->amount,
                $value->name,
                $value->title,
                $value->description,
                $value->payed_at,
            )
        );
    }

    $html .= theme('table', array(
        'header' => $header,
        'rows' => $rows,
        'caption' => t('Költségek listája'),
        'sticky' => TRUE,
        'empty' => t('Még nincsenek Költségek létrehozva.'),
    ));

    $html .= theme('pager', array('tags' => array()));

    return $html;
}

/**
 * Defines the form main for Cost.
 *
 * @return $form
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_cost_base_form() {

    // Values for entity Account
    $select_accounts = db_select('ownership_accounts', 'oa');
    $select_accounts->fields('oa', array('aid', 'name'));
    $select_accounts->orderBy('name', 'ASC');

    $account_entries = $select_accounts->execute()->fetchAll();

    foreach ($account_entries as $a_entry) {
        $account_options[$a_entry->aid] = t($a_entry->name);
    }


    $form = array();

    $form['overview'] = array(
        '#prefix' => '<p>',
        '#suffix' => '</p>',
    );

    $form['amount'] = array(
        '#title' => t('Összeg'),
        '#description' => t('A költség összege'),
        '#type' => 'textfield',
        '#required' => TRUE,
    );

    $form['account'] = array(
        '#title' => t('Folyószámla'),
        '#description' => t('A folyószámla neve'),
        '#type' => 'select',
        '#options' => $account_options,
        '#required' => TRUE,
    );

    $form['title'] = array(
        '#title' => t('Cím'),
        '#description' => t('A költség jogcíme'),
        '#type' => 'textfield',
        '#required' => TRUE,
    );

    $form['description'] = array(
        '#title' => t('Leírás'),
        '#description' => t('A költség leírása'),
        '#type' => 'textarea',
        '#required' => TRUE,
    );

    $form['payed'] = array(
        '#title' => t('Dátum'),
        '#description' => t('A fizetés dátuma'),
        '#type' => 'date_popup',
        '#date_format' => 'Y.m.d.',
        '#required' => TRUE,
    );

    $form['submit'] = array(
        '#type' => 'submit',
        '#value' => t('Költség rögzítése'),
    );

    return $form;
}

/**
 * Implements hook_form().
 * Defines form for creating Cost
 * @params $form id
 * @params &$form_state form values
 * @return $form
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_cost_create_form($form, &$form_state) {
    $form = ownership_cost_base_form();
    $form['overview']['#markup'] = t('Ezen a felületen lehet létrehozni új Költésgeket.');
    return $form;
}

/**
 * Validates form submit for creating Cost.
 * @params $form id
 * @params &$form_state form values
 * @return validated form values
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_cost_create_form_validate($form, &$form_state) {
    $title = $form_state['values']['title'];

    $select_cost_title = db_select('ownership_costs', 'oc')
            ->fields('oc', array('title'))
            ->condition('oc.title', $title)
            ->range(0, 1)
            ->execute();
    $cost_title = $select_cost_title->fetchField();

    if (!empty($cost_title)) {
        form_set_error('title', t('Ilyen címen már létezik költség a rendszerben.'));
    }
}

/**
 * Submits form for creating Cost.
 * @params $form id
 * @params &$form_state form values
 * @return redirect
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_cost_create_form_submit($form, &$form_state) {
    global $user;
    date_default_timezone_set('Europe/Budapest');

    $cost_fields = array(
        'amount' => $form_state['values']['amount'],
        'aid' => (int) trim($form_state['values']['account']),
        'uid' => $user->uid,
        'title' => trim($form_state['values']['title']),
        'description' => trim($form_state['values']['description']),
        'created_at' => date("Y-m-d h:i:s"),
        'payed_at' => $form_state['values']['payed'],
    );

    $cost_id = db_insert('ownership_costs')->fields($cost_fields)->execute();

    $form_state['redirect'] = array('admin/config/ownership/cost',
        array(
            'query' => array(),
            'fragment' => '',
        ),
    );

    drupal_set_message(t('<em>@title</em> címen költség létrehozva.', array('@title' => $cost_fields['title'])));

    // Save Watchdog notice.
    watchdog('ownership', '@title címen költség létrehozva.', array('@title' => $cost_fields['title'], WATCHDOG_NOTICE));
}
