<?php

/**
 * @file Ownership Module functions for Accounts.
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */

/**
 * Accounts listing.
 * @return type html table of Accounts.
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_account_list() {
    $html = '';
    
    $account_priority_options = array(
        0 => t('Profitból számított'),
        1 => t('Bevételből számított'),
        2 => t('Áfa'),
        3 => t('Termék profitból'),
    );

    $header = array(
        array('data' => 'aid', 'field' => 'aid', 'sort' => 'asc'),
        array('data' => t('Folyószámla neve'), 'field' => 'name'),
        array('data' => t('Hányad'), 'field' => 'percent'),
        array('data' => t('Prioritás'), 'field' => 'priority'),
        array('data' => t('Típus'), 'field' => 'type'),
        array('data' => t('Aktív'), 'field' => 'is_active'),
        array('data' => t('User ID'), 'field' => 'uid'),
        array('data' => t('Műveletek')),
    );

    $result = db_select('ownership_accounts', 'a')
            ->extend('PagerDefault')
            ->limit(25)
            ->extend('TableSort')
            ->orderByHeader($header)
            ->fields('a')
            ->execute();

    $rows = array();

    foreach ($result as $value) {
        $links = l(t('Szerkesztés'), "admin/config/ownership/account/edit/$value->aid") .
                '&nbsp;&nbsp;' .
                l(t('Törlés'), "admin/config/ownership/account/delete/$value->aid") .
                '&nbsp;&nbsp;';

        $rows[] = array(
            'data' => array(
                $value->aid,
                $value->name,
                $value->percent,
                $account_priority_options[$value->priority],
                $value->type == 0 ? t('Százalékosan') : t('Összegezve'),
                $value->is_active == 0 ? t('Nem') : t('Igen'),
                $value->uid,
                $links,
            )
        );
    }

    $html .= theme('table', array(
        'header' => $header,
        'rows' => $rows,
        'caption' => t('Számlák listája'),
        'sticky' => TRUE,
        'empty' => t('Még nincsenek Számlák létrehozva.'),
    ));

    $html .= theme('pager', array('tags' => array()));

    return $html;
}

/**
 * Defines the form main for Account.
 *
 * @return $form
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_account_base_form() {

    // Values for entity Account
    $account_type_options = array(
        0 => t('Százalékosan'),
        1 => t('Összegezve'),
    );

    $account_priority_options = array(
        0 => t('Profitból számított'),
        1 => t('Bevételből számított'),
        2 => t('Áfa'),
        3 => t('Termék profitból'),
    );

    $products = ownership_get_membership_products_list();

    $form = array();

    $form['overview'] = array(
        '#prefix' => '<p>',
        '#suffix' => '</p>',
    );

    $form['name'] = array(
        '#title' => t('Számla'),
        '#description' => t('A számla neve'),
        '#type' => 'textfield',
        '#required' => TRUE,
    );

    $form['percent'] = array(
        '#title' => t('Hányad'),
        '#description' => t('Megmutatja a tulajdonosi hányadot. Pl.: 16.67'),
        '#type' => 'textfield',
        '#maxlength' => 5,
        '#required' => TRUE,
    );

    $form['type'] = array(
        '#title' => t('Típus'),
        '#description' => t('A számla típusa'),
        '#type' => 'select',
        '#options' => $account_type_options,
        '#required' => TRUE,
    );

    $form['priority'] = array(
        '#title' => t('Számítási alap'),
        '#description' => t('A hányad melyik alapból számítja az összeget.'),
        '#type' => 'select',
        '#options' => $account_priority_options,
        '#required' => TRUE,
    );

    $form['is_active'] = array(
        '#title' => t('Aktív'),
        '#description' => t('Megmutatja, hogy a számla aktív-e, vagy sem'),
        '#type' => 'checkbox',
    );
    
    $form['source_box'] = array(
        '#type' => 'fieldset',
        '#title' => t('Termékre szűkítés'),
        '#description' => t('Ha van kipipált termék, akkor csak az abból eredő bevételt veszi alapul.'),
        '#collapsible' => TRUE,
        '#collapsed' => FALSE,
    );

    $form['source_box']['source'] = array(
        '#title' => t('Tagság és előfizetés'),
        '#type' => 'checkboxes',
        '#options' => $products,
    );

    $form['submit'] = array(
        '#type' => 'submit',
        '#value' => t('Számla mentése'),
    );

    return $form;
}

/**
 * Implements hook_form().
 * Defines form for creating Account
 * @params $form id
 * @params &$form_state form values
 * @return $form
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_account_create_form($form, &$form_state) {
    $form = ownership_account_base_form();
    $form['overview']['#markup'] = t('Ezen a felületen lehet létrehozni új Folyószámlákat.');
    return $form;
}

/**
 * Validates form submit for creating Account.
 * @params $form id
 * @params &$form_state form values
 * @return validated form values
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_account_create_form_validate($form, &$form_state) {
    $name = $form_state['values']['name'];
    $percent = $form_state['values']['percent'];

    $select_account_name = db_select('ownership_accounts', 'oa')
            ->fields('oa', array('name'))
            ->condition('oa.name', $name)
            ->range(0, 1)
            ->execute();
    $account_name = $select_account_name->fetchField();

    if (!empty($account_name)) {
        form_set_error('name', t('Ilyen nevű folyószámla már létezik termék a rendszerben.'));
    }

    if (!is_numeric($percent) && ($percent >= 0)) {
//    if (!((preg_match('{^[0-9]{1,9}$}', $percent)) && ($percent >= 0))) {
//    if ((filter_var($percent, FILTER_VALIDATE_FLOAT) || filter_var($percent, FILTER_VALIDATE_INT)) && ($percent >= 0)) {
        form_set_error('price', t('Helytelen összeg. Az százalék mezőben pozitív számot adj meg, ami lehet decimális, de a többi hasonló számlával együtt összesen 100%-ot kell adjon.'));
    }
}

/**
 * Submits form for creating Account.
 * @params $form id
 * @params &$form_state form values
 * @return redirect
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_account_create_form_submit($form, &$form_state) {
    global $user;
    
    $source = $form_state['values']['source'];
    foreach($source as $k => $v) {
        if ($v != 0) {
            $v = 1;
        }
    }

    $account_fields = array(
        'name' => trim($form_state['values']['name']),
        'percent' => (double) str_replace(',', '.', $form_state['values']['percent']),
        'type' => (int) trim($form_state['values']['type']),
        'is_active' => (int) trim($form_state['values']['is_active']),
        'products' => !empty($form_state['values']['source']) ? serialize($source) : null,
        'uid' => $user->uid,
    );

    $account_id = db_insert('ownership_accounts')->fields($account_fields)->execute();

    $form_state['redirect'] = array('admin/config/ownership/account',
        array(
            'query' => array(),
            'fragment' => '',
        ),
    );

    drupal_set_message(t('<em>@name</em> nevű folyószámla létrehozva.', array('@name' => $account_fields['name'])));

    // Save Watchdog notice.
    watchdog('ownership', '@name nevű folyószámla létrehozva.', array('@name' => $account_fields['name'], WATCHDOG_NOTICE));
}

/**
 * Implements hook_form().
 * Defines form for editing Account
 * @params $form id
 * @params &$form_state form values
 * @params $aid account id
 * @return $form
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_account_edit_form($form, &$form_state, $aid) {
    $account = db_select('ownership_accounts', 'oa')->fields('oa')->condition('aid', $aid)->execute()->fetchObject();

    $form = ownership_account_base_form();
    $form['overview']['#markup'] = t('Ezen a felületen lehet szerkeszteni a meglévő folyószámlákat.');
    $form['name']['#default_value'] = $account->name;
    $form['percent']['#default_value'] = $account->percent;
    $form['priority']['#default_value'] = $account->priority;
    $form['type']['#default_value'] = $account->type;
    $form['is_active']['#default_value'] = $account->is_active;
    if ($account->products != null) {
        $form['source_box']['source']['#default_value'] = unserialize($account->products);
    }

    return $form;
}

/**
 * Validates form submit for editing Account.
 * @params $form id
 * @params &$form_state form values
 * @return validated form values
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_account_edit_form_validate($form, &$form_state) {
    $percent = $form_state['values']['percent'];

    if (!is_numeric($percent) && ($percent >= 0)) {
//    if (!((preg_match('{^[0-9]{1,9}$}', $percent)) && ($percent >= 0))) {
//    if ((filter_var($percent, FILTER_VALIDATE_INT) || filter_var($percent, FILTER_VALIDATE_FLOAT)) && ($percent >= 0)) {
        form_set_error('price', t('Helytelen összeg. Az százalék mezőben pozitív számot adj meg, ami lehet decimális, de a többi hasonló számlával együtt összesen 100%-ot kell adjon.'));
    }
}

/**
 * Submits form for edited Account.
 * @params $form id
 * @params &$form_state form values
 * @return redirect
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_account_edit_form_submit($form, &$form_state) {
    global $user;
    $aid = $form_state['build_info']['args'][0];

    $account_fields = array(
        'name' => $form_state['values']['name'],
        'percent' => (double) str_replace(',', '.', $form_state['values']['percent']),
        'type' => $form_state['values']['type'],
        'priority' => $form_state['values']['priority'],
        'is_active' => $form_state['values']['is_active'],
        'products' => !empty($form_state['values']['source']) ? serialize($form_state['values']['source']) : null,
        'uid' => !empty($user->uid) ? $user->uid : 0,
    );

    $account_updated = db_update('ownership_accounts')->fields($account_fields)->condition('aid', $aid, '=')->execute();

    //Redirect ti the listing site.
    $form_state['redirect'] = array('admin/config/ownership/account',
        array(
            'query' => array(),
            'fragment' => '',
        ),
    );

    // Message for the User.
    drupal_set_message(t('<em>@name</em> nevű folyószámla adatai sikeresen módosításra kerültek.', array('@name' => $account_fields['name'])));


    // Save to Watchdog notice.
    watchdog('ownership', '@name nevű folyószámla adatai sikeresen megváltoztak.', array('@name' => $account_fields['name'], WATCHDOG_NOTICE));
}

/**
 * Implements hook_form().
 * Creates form for confirm deleting Account.
 * @params $form id
 * @params &$form_state form values
 * @params $aid account id
 * @return redirect
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_account_delete_form($form, &$form_state, $aid) {
    $result = db_select('ownership_accounts', 'oa')->fields('oa', array('name'))->condition('oa.aid', $aid)->range(0, 1)->execute()->fetchField();

    $form = array();
    $form['aid'] = array(
        '#type' => 'value',
        '#value' => $aid,
    );

    // Keep the old Account name for the Watchdog notice
    $form['name'] = array(
        '#type' => 'hidden',
        '#value' => $result,
    );

    return confirm_form(
            $form, t('<em>"' . $result . '"</em> biztosan törölhető?'), 'admin/config/ownership/account', t('Ez a művelet nem vonható vissza.'), t('Folyószámla törlése'), t('Mégsem')
    );
}

/**
 * Submit for deleting Account.
 * @params $form id
 * @params &$form_state form values
 * @params $aid account id
 * @return redirect
 * @author Ferentzi, Sándor <sandor.ferentzi@gmail.com>
 */
function ownership_account_delete_form_submit($form, &$form_state) {
    $aid = array('name' => $form_state['values']['aid']);
    $name = $form_state['values']['name'];

    $account_deleted = db_delete('ownership_accounts')->condition('aid', $aid, '=')->execute();

    // Redirect to the listing page
    $form_state['redirect'] = array('admin/config/ownership/account',
        array(
            'query' => array(),
            'fragment' => '',
        ),
    );

    // Message for the User.
    drupal_set_message(t('A Folyószámla sikeresen törölve.'));

    // Save Watchdog notice.
    watchdog('ownership', '@name nevű folyószámla sikeresen törölve.', array('@name' => $name), WATCHDOG_NOTICE);
}
