<?php

/**
 * @file The global functions for the Certify module.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */

/**
 * Check if a value is an integer.
 * @param mixed $variable The value.
 * @return boolean Returns TRUE if the value is an integer else returns FALSE.
 * @author Müller Péter <egyeves@gmail.com>
 */
function _membership_is_int($variable) {
    return ctype_digit(strval($variable));
}

/**
 * Get the fields form the certify_candidate_groups table.
 * @return string The name field value.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_get_user_fields($uid) {
    return db_query_range('SELECT uid, name, mail FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid,))->fetchAssoc();
}

/**
 * Easy look debug function.
 * @params type $var array.
 * @params type $json boolean flag.
 * @return string Formatted array tree.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function mdd($var, $json = FALSE) {
    if ($json) {
        header('Content-Type: application/json');
        echo drupal_json_output($var);
        exit();
    } else {
        echo '<pre>';
        print_r($var);
        echo '</pre>';
    }
}

/**
 * Filter the array if the value is '' or null or false give back empty string;
 * @param type $form_state It has to contain [values] associative array
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_array_filter_empty_value($value, $form_state) {
    if (is_array($value) && isset($form_state['values']['array_filter'])) {
        $value = array_keys(array_filter($value));
        return $value;
    }
    return $value;
}

/**
 * Price value formatter.
 * @param type number.
 * @return string Formatted.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_format_price($number) {
    return number_format($number, 0, ',', ' ') . ' ' . variable_get('membership_firm_invoice_currency', 'Ft');
}

/**
 * Check if a string is serialized
 * @param string $string
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_is_serial($string) {
    return (@unserialize($string) !== false);
}

/**
 * Check if a string is serialized
 * @param string $bigger
 * @param string $smaller
 * @return absolute value number.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_get_abs_different($bigger, $smaller) {
    return abs(floatval($bigger) - floatval($smaller));
}

/**
 * Check if the uid is valid.
 * @param string $uid The hash code.
 * @return boolean FALSE if the uid is not valid.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_isset_uid($uid) {
    $result = db_query_range('SELECT count(uid) uid FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid))->fetchField();
    return (empty($result)) ? FALSE : TRUE;
}

/**
 * Adds a new role
 * @machine_name - the name of the role to be added
 * @weight - the optional display weight for role
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_add_role($machine_name, $weight = 0) {
    $role = new stdClass();
    $role->name = $machine_name;
    $role->weight = $weight;
    user_role_save($role);
}

/**
 * Adds permissions to a role
 * @role_name - the name of the role to receive an update
 * @permissions - the array of permissions to add
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_add_permissions($role_name, $permissions) {
    $role = user_role_load_by_name($role_name);
    user_role_grant_permissions($role->rid, $permissions);
}

/**
 * Return the two digits string
 * @params type integer.
 * @return string.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_two_digits($int) {
    if (strlen($int) == 1) {
        return '0' . $int;
    } else {
        return $int;
    }
}

/**
 * Add time value to the given date.
 * @params type string date.
 * @params type string.
 * @params type integer.
 * @return timestamp.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_add_time($date, $type = 'days', $int = 1) {

    $new_date = strtotime("+" . $int . " " . $type, strtotime($date));
    return $new_date;
}

/**
 * Show full name in set order.
 * @params type string $lastname.
 * @params type string $firstname.
 * @return string Fullname.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_show_name($lastname, $firstname) {
    if (variable_get('membership_basic_name', '1') == 1) {
        return _capitalize($firstname) . ' ' . _capitalize($lastname);
    } else {
        return _capitalize($lastname) . ' ' . _capitalize($firstname);
    }
}

/**
 * Egy e-mailt küld a rendszerből a felhasználónak.
 *
 * @param string $name
 * A címzett neve.
 *
 * @param type string $email.
 * A címzett e-mail címe.
 *
 * @param type string $subject.
 * Az e-mail tárgya.
 *
 * @param string $message
 * Az e-mail üzetnet.
 *
 * @author Müller Péter <egyeves@gmail.com>
 */
function _membership_email_send($name, $email, $subject, $message) {

    $settings_mail_from_name = variable_get('certify_email_sender_name');
    $settings_mail_from_email = variable_get('certify_email_sender_email');

    $to = '"' . '=?UTF-8?B?' . base64_encode($name) . '?=' . '" <' . $email . '>';

    $header = 'MIME-Version: 1.0' . PHP_EOL;
    $header .= 'Content-type: text/plain; format=flowed; charset="UTF-8"' . PHP_EOL;
    $header .= 'Content-Transfer-Encoding: 8bit' . PHP_EOL;
    $header .= 'From: "' . '=?UTF-8?B?' . base64_encode($settings_mail_from_name) . '?=' . '" <' . $settings_mail_from_email . '>' . PHP_EOL;
    $header .= 'Reply-To: ' . $settings_mail_from_email . PHP_EOL;
    $header .= 'Return-Path: ' . $settings_mail_from_email . PHP_EOL;
    $header .= 'X-Mailer: PHP-' . phpversion() . PHP_EOL;

    mail($to, '=?UTF-8?B?' . base64_encode($subject) . '?=', $message, $header);
}

/**
 * Get period length in days.
 * @param type object $product
 * @return timestamp.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_period_deadline($product, $start) {

    // Get data form database
    $deadline = $start;
    switch ($product->period_type) {
        case 0:
            $deadline = _membership_add_time(date('Y-m-d H:i:s', strtotime($start)), 'days', (int) $product->period_length);
            break;
        case 1:
            $deadline = _membership_add_time(date('Y-m-d H:i:s', strtotime($start)), 'week', (int) $product->period_length);
            break;
        case 2:
            $deadline = _membership_add_time(date('Y-m-d H:i:s', strtotime($start)), 'month', (int) $product->period_length);
            break;
        case 3:
            $deadline = _membership_add_time(date('Y-m-d H:i:s', strtotime($start)), 'year', (int) $product->period_length);
            break;
        default:
            $deadline = null;
    }

    return $deadline;
}

/**
 * Get all of order details.
 * @param type integer $oid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_order_get_details($oid) {

    // Get data form database
    $query = db_select('membership_orders', 'o')
            ->condition('o.oid', $oid)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid']);
    $order = $query->execute()->fetchAssoc();

    return $order;
}

/**
 * Get all of order details.
 * @param type integer $oid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_upsell_order_get_details($oid) {

    // Get data form database
    $query = db_select('membership_orders', 'o')
            ->condition('o.uoid', $oid)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid']);
    $order = $query->execute()->fetchAssoc();

    return $order;
}

/**
 * Get order details object.
 * @param integer $iid
 * @param string $column
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_invoice_get($iid, $column = 'iid') {

    // Get data form database
    $query = db_select('membership_invoices', 'i')
            ->condition('i.' . $column, $iid)
            ->fields('i');
    $invoice = $query->execute()->fetchObject();

    return $invoice;
}

/**
 * Get order details object.
 * @param integer $oid
 * @param integer $order_type [0,1]
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_invoice_get_object($oid, $order_type) {

    // Get data form database
    $query = db_select('membership_invoices', 'i')
            ->condition('i.oid', $oid)
            ->condition('i.order_type', $order_type)
            ->fields('i');
    $invoice = $query->execute()->fetchObject();

    return $invoice;
}

/**
 * Get all of order details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_order_get_all_details_object($oid) {

    // Get data form database
    $query = db_select('membership_orders', 'o')
            ->condition('o.oid', $oid)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid', 'commission', 'vat']);
    $order = $query->execute()->fetchObject();

    return $order;
}

/**
 * Get user's last order.
 * @param type integer $uid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_user_last_order_object($uid) {

    // Get data form database
    $query = db_select('membership_orders', 'o')
            ->condition('o.id', $uid)
            ->fields('o')
            ->orderBy('created', 'DESC');
    $query->leftJoin('membership_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid', 'commission', 'vat']);
    $query->range(0, 1);
    $order = $query->execute()->fetchObject();

    return $order;
}

/**
 * Get order details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_order_get_details_object($oid) {

    // Get data form database
    $query = db_select('membership_orders', 'o')
            ->condition('o.oid', $oid)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_period_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid']);
    $order = $query->execute()->fetchObject();

    return $order;
}

/**
 * Get period promotion details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_promo_get_details_object($promo_id) {

    // Get data form database
    $query = db_select('membership_period_promotions', 'o')
            ->condition('o.promo_id', $promo_id)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_period_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid']);
    $order = $query->execute()->fetchObject();

    return $order;
}

/**
 * Get period promotion details object.
 * @param type integer $poid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_promo_get_details_by_poid($poid) {

    // Get data form database
    $query = db_select('membership_period_promotions', 'o')
            ->condition('o.poid', $poid)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_period_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid']);
    $order = $query->execute()->fetchObject();

    return $order;
}

/**
 * Get period promotion details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_promo_get_details_assoc($promo_id) {

    // Get data form database
    $query = db_select('membership_period_promotions', 'o')
            ->condition('o.promo_id', $promo_id)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_period_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid']);
    $order = $query->execute()->fetchAssoc();

    return $order;
}

/**
 * Get period promotion details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_order_get_details_object($poid) {

    // Get data form database
    $query = db_select('membership_period_orders', 'o')
            ->condition('o.poid', $poid)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_period_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid']);
    $order = $query->execute()->fetchObject();

    return $order;
}

/**
 * Get period promotion details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_order_get_details_assoc($poid) {

    // Get data form database
    $query = db_select('membership_period_orders', 'o')
            ->condition('o.poid', $poid)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_period_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid']);
    $order = $query->execute()->fetchAssoc();

    return $order;
}

/**
 * Get email object.
 * @param integer $eid
 * @param string $column
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_email_object($eid, $column = 'eid') {

    // Get data form database
    $query = db_select('membership_emails', 'e')
            ->condition('e.' . $column, $eid)
            ->fields('e');
    $email = $query->execute()->fetchObject();

    return $email;
}

/**
 * Get email array.
 * @param integer $eid
 * @param string $column
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_email_assoc($eid, $column = 'eid') {

    // Get data form database
    $query = db_select('membership_emails', 'e')
            ->condition('e.' . $column, $eid)
            ->fields('e');
    $email = $query->execute()->fetchAssoc();

    return $email;
}

/**
 * Get email list.
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_email_list() {

    // Get data form database
    $query = db_select('membership_emails', 'e')
            ->fields('e');
    $emails = $query->execute()->fetchAll();

    return $emails;
}

/**
 * Close period order.
 * @param type number $promo_id
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_order_close($poid) {

    $update = db_update('membership_period_orders')
            ->fields(['closed' => 1])
            ->condition('poid', $poid)
            ->execute();

    return $update;
}

/**
 * Close period promotion.
 * @param type number $promo_id
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_order_full_close($poid) {

    $order = membership_period_order_get_details_assoc($poid);
    $product = membership_period_product_get_details_object($order['pid']);

    $dropping = membership_period_order_get_dropping($poid);
    membership_period_order_manipulate_roles($order['uid'], $product, $dropping, 'remove');

    $update = db_update('membership_period_orders')
            ->fields(['closed' => 1])
            ->condition('poid', $poid)
            ->execute();

    return $update;
}

/**
 * Period order create.
 * @param type array $fields order fields.
 * @param type number $poid order id.
 * @return mixed Order object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_order_merge($fields, $poid = null, $assoc = false) {
    // Base fields
    $order = membership_period_order_get_details_object($poid);
    if (empty($order)) {
        $product = membership_period_product_get_details_object($fields['pid']);
        $start = date('Y-m-d H:i:s', _membership_add_time(date('Y-m-d') . ' 00:00:00', 'days', (int) $product->promotime));
        $deadline = membership_get_period_deadline($product, $start);
        $end = $deadline !== null ? date('Y-m-d H:i:s', $deadline) : $deadline;
        $base = array(
            'id' => $fields['id'],
            'pid' => $fields['pid'],
            'created' => REQUEST_TIME,
            'start_at' => isset($fields['start_at']) ? $fields['start_at'] : $start,
            'end_at' => isset($fields['end_at']) ? $fields['end_at'] : $end,
            'ip' => isset($fields['ip']) ? $fields['ip'] : _get_real_ip(),
            'price' => isset($fields['price']) ? $fields['price'] : $product->price,
            'shipping' => isset($fields['shipping']) ? $fields['shipping'] : 0,
            'vat' => isset($fields['vat']) ? $fields['vat'] : $product->vat,
            'is_activated' => 0,
            'is_sent' => isset($fields['is_sent']) ? $fields['is_sent'] : 3,
            'is_invoiced' => 0,
            'payment_id' => isset($fields['payment_id']) ? $fields['payment_id'] : null,
            'token' => isset($fields['token']) ? $fields['token'] : null,
            'thank_you' => isset($fields['thank_you']) ? $fields['thank_you'] : $product->thank_you,
        );
        $poid = db_insert('membership_period_orders')->fields($base)->execute();
    } else {
        db_merge('membership_period_orders')->key('poid', $poid)->fields($fields)->execute();
    }
    if ($assoc) {
        return membership_period_order_get_details_assoc($poid);
    } else {
        return membership_period_order_get_details_object($poid);
    }
}

/**
 * Activate order after success payment.
 * @param type integer $oid Order id
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_order_activate($oid, $cron = FALSE) {

    // Megvizsgáljuk a megadott érték pozitív egész-e.
    if (!is_numeric($oid) || $oid < 1 || $oid != round($oid)) {
        // Save to watchdog.
        watchdog('membership', t('A rendelés azonosítónak pozitív egész számnak kell lennie. Oid: %oid'), array('%oid' => $oid, WATCHDOG_NOTICE));

        return FALSE;
    } else {
        // Ha pozitív egész, akkor megnézzük, hogy létezik-e a rendelés azonosító.
        $order = membership_period_order_get_details_object($oid);
        if (empty($order)) {
            // Save to watchdog.
            watchdog('membership', t('Ilyen sorszámmal nem létezik rendelés a rendszerben. Period order id: %oid'), array('%oid' => $oid, WATCHDOG_NOTICE));

            return FALSE;
        } else {
            if ($order->is_activated) {
                // Save to watchdog.
                watchdog('membership', t('Ez a megrendelés már aktiválva van. Period order id: %poid'), array('%poid' => $oid, WATCHDOG_NOTICE));

                return FALSE;
            }
        }
    }

    // Aktiváljuk a megrendelést
    db_update('membership_period_orders')
            ->condition('poid', $order->poid)
            ->fields(['is_activated' => REQUEST_TIME])
            ->execute();

    $uid = $order->uid;
    $username = $order->lastname . ' ' . $order->firstname;
    $email = $order->email;
    $pid = $order->pid;
    
    // Számla elkészítése
    $max = db_query('SELECT MAX(invoice_number) FROM {membership_invoices}')->fetchField();

    if ($order->is_firm_invoice == 0) {
        // Normál számla
        $name = "$order->lastname $order->firstname";
        $zip = $order->zip;
        $city = $order->city;
        $address = $order->address;
    } else {
        // Céges számla.
        $name = $order->firm_name;
        $zip = $order->firm_zip;
        $city = $order->firm_city;
        $address = $order->firm_address;
    }

    if ($order->country != 'HU') {
        // Ha nem magyarországi cím, akkor kiírjuk azt ország nevét is.
        $country = db_query('SELECT * FROM {membership_country} ORDER BY name ASC')->fetchAllKeyed();
    } else {
        $country = ['HU' => 'HU'];
    }

    $invoice = array(
        'id' => $order->id,
        'oid' => $order->poid,
        'order_type' => 1,
        'created' => REQUEST_TIME,
        'issued' => REQUEST_TIME,
        'invoice_number' => $max + 1,
        'quantity' => 1,
        'amount' => (int) trim($order->price),
        'is_firm_invoice' => $order->is_firm_invoice,
        'cancellation' => NULL,
        'error' => '',
        'name' => $name,
        'zip' => $zip,
        'city' => $city,
        'address' => $address,
        'country' => $country[$order->country],
        'deadline' => REQUEST_TIME,
        'product' => $order->proforma_name,
        'hash' => _make_hash(32),
        'mode' => 2,
        'vat' => $order->vat,
    );

    if ($invoice['amount'] != 0) {
        // Ha nagyobb a termnék értéke mint 0.
        $iid = db_insert('membership_invoices')->fields($invoice)->execute();

        // Létrehozzuk a számlát.
        membership_create_invoice($iid);

        // Előkésztjük az orders táblában a frissítendő mezőket.
        $order_update = array(
            'price' => $invoice['amount'],
            'is_activated' => REQUEST_TIME,
            'is_invoiced' => $iid,
        );
    } else {
        // Ha a számla végösszege nulla, azt jelenti, hogy a felhasználó nem fizet
        // a termékért, ezért a is_invoiced mezőt nem kell módostíani, különben
        // hibaüznetet kapunk, mert a kiszámolázott számlára mutat, aminek az
        // azonosítója nem lehet nulla.
        $order_update = array(
            'price' => $invoice['amount'],
            'is_activated' => REQUEST_TIME,
        );
    }

    $num_updated = db_update('membership_period_orders')->fields($order_update)->condition('poid', $invoice['oid'], '=')->execute();

    if (!$uid && !$cron) {
        // Ha még nem aktivált felhasználó létrehozzunk az adatbázisban
        // és összekötjük a megerendelési adatokkal.
        $result = create_new_user($username, $email);

        $uid = $result['account']->uid;
        $num_updated = db_update('membership_users')->fields(array('uid' => $uid,))->condition('id', $invoice['id'], '=')->execute();

        // E-mail küldés az belépési adatokról.
        _new_user_email_send($username, $email, $result['username'], $result['password']);
    }

    // A szerepkörök, és hírvelek ki-be kikapcsolása.
    $product = membership_period_product_get_details_object($pid);
    
    // Ha van aktív kiváltandó rendelése azt lezárjuk
    $replace_products = unserialize($product->replace_products);
    if ($replace_products != false) {
        db_update('membership_period_orders')
            ->fields(['closed' => 1])
            ->condition('pid', array_values($replace_products), 'IN')
            ->condition('is_activated', 0, '>')
            ->condition('closed', 0)
            ->execute();
    } 


    // A szerepkörök törlése és hozzáadása.
    $dropping = membership_period_order_get_dropping($order->poid);
    membership_period_order_manipulate_roles($uid, $product, $dropping);

    // Felhasználó törlése és hozzáadása a hírlevél listákhoz.
    //
      // E-mail küldés az aktiválásról
    if (!$cron) {
        _activate_email_send($username, $email, $product->name);
    } else {
        _period_order_renew_email_send($username, $email, $product->name);
    }

    // A felhasználó felcímkézése az IM-ben.
    // Az e-mailhez hozzáadjuk az aktivált címkét és eltávolítjuk a megrendelt és
    // lejárt címkéket. Ha a felhasználó nem létezik, nem történik semmi.
    $user_data = db_query_range('SELECT lastname, firstname, email, hash, phone FROM {membership_users} WHERE id = :id', 0, 1, array(':id' => $order->id))->fetchObject();

    membership_period_order_manipulate_ac($user_data, $product, $dropping);

    // Ha az alert modul be van kapcsolva megnézi, hogy az illető rendelése
    // szerepel-e az alert_nopayers vagyis nem fizetők táblájában.
    if (module_exists('alert')) {
        $nonpayer = db_query_range('SELECT oid FROM {alert_nonpayers} WHERE oid = :oid', 0, 1, array(':oid' => $invoice['oid']))->fetchField();
        if (!empty($nonpayer)) {
            // Ha szerepel a táblában töröljük, hiszen az aktiválás bizonyítja, hogy
            // fizetett.
            db_delete('alert_nonpayers')->condition('oid', $nonpayer, '=')->execute();
        }
    }

    return TRUE;
}

/**
 * Get period promotion details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_promo_used_by_user($pid) {

    // Get data form database
    $query = db_select('membership_period_promotions', 'p')
            ->condition('p.pid', $pid)
            ->fields('p');
    $order = $query->execute()->fetchObject();

    if (!empty($order)) {
        return true;
    }
    return false;
}

/**
 * Get product details object.
 * @param integer $pid
 * @param string $product_type
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_any_product_object($pid, $product_type) {

    // Get data form database
    if ($product_type == 'period') {
        $product = membership_period_product_get_details_object($pid);
    } else {
        $product = membership_product_get_details_object($pid);
    }

    return $product;
}

/**
 * Get product details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_product_get_details_object($pid) {

    // Get data form database
    $query = db_select('membership_products', 'p')
            ->condition('p.pid', $pid)
            ->fields('p');
    $product = $query->execute()->fetchObject();

    return $product;
}

/**
 * Get period product details object.
 * @param type integer $pid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_product_get_details_object($pid) {

    // Get data form database
    $query = db_select('membership_period_products', 'p')
            ->condition('p.pid', $pid)
            ->fields('p');
    $product = $query->execute()->fetchObject();

    return $product;
}

/**
 * Get period product details object.
 * @param type integer $pid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_product_get_details_assoc($pid) {

    // Get data form database
    $query = db_select('membership_period_products', 'p')
            ->condition('p.pid', $pid)
            ->fields('p');
    $product = $query->execute()->fetchAssoc();

    return $product;
}

/**
 * Get period product details object.
 * @param type integer $pid
 * @return number.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_order_get_dropping($poid) {

    // Get data form database
    $query = db_select('membership_period_orders', 'o')
            ->condition('o.poid', $poid)
            ->fields('o');
    $order = $query->execute()->fetchObject();

    $query = db_select('membership_period_orders', 'o')
            ->condition('o.pid', $order->pid)
            ->condition('o.id', $order->id)
            ->condition('o.is_activated', 0, '!=')
            ->fields('o');
    $dropping = (int) $query->execute()->rowCount() - 1;

    return $dropping;
}

/**
 * Get period product details object.
 * @param type integer $uid
 * @param type object $product
 * @param type integer $dropping
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_order_manipulate_roles($uid, $product, $dropping = 0, $mode = 'add') {

    $roles = unserialize($product->roles);

    if ($mode == 'add') {
        // A szerepkörök törlése és hozzáadása.
        if (isset($roles[$dropping])) {
            $roles_remove = $roles[$dropping]['roles_order_remove'];
            $rids = array();
            foreach ($roles_remove as $role) {
                if ($role != 0) {
                    $rids[] = $role;
                }
            }
            _set_roles($uid, $rids, 'remove_role');

            $roles_add = $roles[$dropping]['roles_order_add'];
            $rids = array();
            foreach ($roles_add as $role) {
                if ($role != 0) {
                    $rids[] = $role;
                }
            }
            _set_roles($uid, $rids, 'add_role');
        }
    } else {
        // A szerepkörök törlése és hozzáadása.
        $roles_remove = $roles[$dropping]['roles_resign_remove'];
        $rids = array();
        foreach ($roles_remove as $role) {
            if ($role != 0) {
                $rids[] = $role;
            }
        }
        _set_roles($uid, $rids, 'remove_role');

        $roles_add = $roles[$dropping]['roles_resign_add'];
        $rids = array();
        foreach ($roles_add as $role) {
            if ($role != 0) {
                $rids[] = $role;
            }
        }
        _set_roles($uid, $rids, 'add_role');
    }

    return true;
}

/**
 * Get period product details object.
 * @param type integer $uid
 * @param type object $product
 * @param type string $mode 'add' or 'revert'
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_promo_manipulate_roles($uid, $product, $mode = 'add') {

    $roles = unserialize($product->roles);

    if ($mode == 'add') {
        // A szerepkörök törlése és hozzáadása.
        $roles_remove = $roles['promo']['roles_order_add'];
        $rids = array();
        foreach ($roles_remove as $role) {
            if ($role != 0) {
                $rids[] = $role;
            }
        }
        _set_roles($uid, $rids, 'add_role');

        $roles_add = $roles['promo']['roles_order_remove'];
        $rids = array();
        foreach ($roles_add as $role) {
            if ($role != 0) {
                $rids[] = $role;
            }
        }
        _set_roles($uid, $rids, 'remove_role');
    } else {
        // A szerepkörök törlése és hozzáadása.
        $roles_remove = $roles['promo']['roles_end_remove'];
        $rids = array();
        foreach ($roles_remove as $role) {
            if ($role != 0) {
                $rids[] = $role;
            }
        }
        _set_roles($uid, $rids, 'remove_role');

        $roles_add = $roles['promo']['roles_end_add'];
        $rids = array();
        foreach ($roles_add as $role) {
            if ($role != 0) {
                $rids[] = $role;
            }
        }
        _set_roles($uid, $rids, 'add_role');
    }

    return true;
}

/**
 * Get period product details object.
 * @param type object $user_data
 * @param type object $product
 * @param type integer $dropping
 * @return mixed.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_order_manipulate_ac($user_data, $product, $dropping = 0) {

    // A felhasználó szinkronizálása active campaign-ban.
    $ac = new ActiveCampaign(variable_get('membership_ac_api_url'), variable_get('membership_ac_api_key'));

    $nls = unserialize($product->newsletters);

    if (isset($nls[$dropping])) {

        $nl_order = $nls[$dropping]['newsletter_order'];

        // Címkék összerakása
        $tags_add = isset($nl_order['order_tags_add']) ? explode(',', $nl_order['order_tags_add']) : [];
        $tags_add[] = 'M_' . $product->table_name; // $product['table_name']
        $tags_remove = isset($nl_order['order_tags_remove']) ? explode(',', $nl_order['order_tags_remove']) : [];

        // Felhazsnáló és címkék szinkronoizálása.
        $contact = array(
            'email' => $user_data->email,
            'first_name' => $user_data->firstname,
            'last_name' => $user_data->lastname,
            'phone' => $user_data->phone,
            'tags' => $tags_add,
        );
        $contact_sync = $ac->api("contact/sync", $contact);

        // Címkék eltávolítása.
        $post_data = array(
            'email' => $user_data->email,
            'tags' => $tags_remove,
        );
        $response = $ac->api("contact/tag_remove", $post_data);

        // Elindítjuk a felhasználót egy automatizmuson.
        $post_data = array(
            'contact_email' => $user_data->email, // include this or contact_id
            'automation' => isset($nl_order['newsletter_order']) ? $nl_order['newsletter_order'] : '', // one or more
        );

        $response = $ac->api("automation/contact/add", $post_data);

        return $response;
    }

    return null;
}

/**
 * Get period product details object.
 * @param type object $user_data
 * @param type object $product
 * @param type integer $dropping
 * @return mixed.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_promo_manipulate_ac($user_data, $product) {

    // A felhasználó szinkronizálása active campaign-ban.
    $ac = new ActiveCampaign(variable_get('membership_ac_api_url'), variable_get('membership_ac_api_key'));

    $nls = unserialize($product->newsletters);

    $nl_order = $nls['promo'];

    // Címkék összerakása
    $tags_add = explode(',', $nl_order['order_tags_add']);
    $tags_add[] = 'M_' . $product->table_name;
    $tags_remove = explode(',', $nl_order['order_tags_remove']);

    // Felhazsnáló és címkék szinkronoizálása.
    $contact = array(
        'email' => $user_data->email,
        'first_name' => $user_data->firstname,
        'last_name' => $user_data->lastname,
        'phone' => $user_data->phone,
        'tags' => $tags_add,
    );
    $contact_sync = $ac->api("contact/sync", $contact);

    // Címkék eltávolítása.
    $post_data = array(
        'email' => $user_data->email,
        'tags' => $tags_remove,
    );
    $response = $ac->api("contact/tag_remove", $post_data);

    // Elindítjuk a felhasználót egy automatizmuson.
    $post_data = array(
        'contact_email' => $user_data->email, // include this or contact_id
        'automation' => $nl_order['newsletter_order'], // one or more
    );

    $response = $ac->api("automation/contact/add", $post_data);

    return $response;
}

/**
 * Close period promotion.
 * @param type number $promo_id
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_period_promo_close($promo_id) {

    $promo = membership_period_promo_get_details_assoc($promo_id);
    $order = membership_period_order_get_details_assoc($promo['poid']);
    $product = membership_period_product_get_details_object($order['pid']);
    membership_period_promo_manipulate_roles($order['uid'], $product, 'remove');

    $update = db_update('membership_period_promotions')
            ->fields(['closed' => 1])
            ->condition('promo_id', $promo_id)
            ->execute();

    return $update;
}

/**
 * Get product details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_product_get_upsell_details_object($pid) {

    // Get data form database
    $query = db_select('membership_upsell_products', 'u')
            ->condition('u.pid', $pid)
            ->fields('u', array('upid', 'status', 'price', 'title', 'description'));
    $query->leftJoin('membership_products', 'p', '(u.upid = p.pid)');
    $query->fields('p');
    $product = $query->execute()->fetchObject();

    return $product;
}

/**
 * Get all of membership user details object.
 * @param type integer $id The user id column in the orders table.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_user_details_object($id) {

    // Get data form database
    // $user = db_query_range('SELECT * FROM {membership_users} WHERE id = :id', 0, 1, array(':id' => $order->id,))->fetchObject();

    $query = db_select('membership_users', 'u')
            ->condition('u.id', $id)
            ->fields('u');
    $user = $query->execute()->fetchObject();

    return $user;
}

/**
 * Get all of membership user details object.
 * @param type integer $uid The user id column in the orders table.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_user_details_object_by_uid($uid) {

    // Get data form database
    // $user = db_query_range('SELECT * FROM {membership_users} WHERE id = :id', 0, 1, array(':id' => $order->id,))->fetchObject();

    $query = db_select('membership_users', 'u')
            ->condition('u.uid', $uid)
            ->fields('u');
    $user = $query->execute()->fetchObject();

    return $user;
}

/**
 * Get all of membership user details object.
 * @param type integer $uid The user id column in the orders table.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_user_details_object_by_email($email) {

    // Get data form database
    // $user = db_query_range('SELECT * FROM {membership_users} WHERE id = :id', 0, 1, array(':id' => $order->id,))->fetchObject();

    $query = db_select('membership_users', 'u')
            ->condition('u.email', check_plain(trim($email)))
            ->fields('u');
    $user = $query->execute()->fetchObject();

    return $user;
}

/**
 * Get all of order details.
 * @param type integer $oid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_user_get_cards($uid) {

    if (module_exists('barion_api')) {
        // Get data form database
        $query = db_select('barion_api_tokens', 'c')
                ->condition('c.uid', $uid)
                ->condition('c.created', '', '!=')
                ->fields('c');
        $tokens = $query->execute()->fetchAll();

        $cards = [null => t('Új bankkártya')];
        foreach ($tokens as $token) {
            $expire = date($token->card_year . '-' . _membership_two_digits((int) $token->card_month) . '-01');
            if ($expire > date('Y-m-d')) {
                $cards[$token->id] = $token->card_type . ' xx' . $token->card_mask . ' (' . _membership_two_digits((int) $token->card_month) . '/' . substr($token->card_year, 2, 2) . ')';
            }
        }

        return $cards;
    }
    return false;
}

/**
 * Get all products list.
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_products_list() {

    // Get data form database
    $query = db_select('membership_products', 'p')
            ->fields('p');
    $products = $query->execute()->fetchAll();

    foreach ($products as $product) {
        $list[$product->pid] = $product->name . ' (' . _membership_format_price($product->price) . ')';
    }

    return $list;
}

/**
 * Get all products list.
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_period_products_list() {

    // Get data form database
    $query = db_select('membership_period_products', 'p')
            ->fields('p');
    $products = $query->execute()->fetchAll();

    foreach ($products as $product) {
        $list[$product->pid] = $product->name . ' (' . _membership_format_price($product->price) . ')';
    }

    return $list;
}

/**
 * Redirect thank you page with analytics params.
 * @param type integer $oid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_redirect_to_thank_you($oid, $thank_you_link = '') {

    if ($thank_you_link == '') {
        $thank_you_link = variable_get('membership_thankyou_link');
    }

    // Get order details
    $order = membership_order_get_details($oid);

    // Átiránytás beállítása, hogy a köszönöm oldalon a Google Analytics tudja rögzíteni az adatokat.
    $country = db_query('SELECT name FROM {membership_country} WHERE code = :code', array(':code' => $order['country']))->fetchField();
    $product_name = $order['name'];
    $product_group_name = db_query('SELECT name FROM {membership_product_groups} WHERE pgid = :pgid', array(':pgid' => $order['pgid']))->fetchField();

    $query = [
        'transid' => $oid,
        'store' => variable_get('membership_site_name'),
        'total' => $order['price'],
        'tax' => $order['price'] * $order['vat'] / 100,
        'shipping' => $order['shipping'],
        'city' => $order['city'],
        'state' => '',
        'country' => $country,
        'sku' => $order['pid'],
        'name' => $product_name,
        'category' => $product_group_name,
        'email' => $order['email'],
        'muid' => $order['id'],
    ];
    drupal_goto($thank_you_link, array('query' => $query));
}

/**
 * Redirect to product order form.
 * @param type integer $pid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_redirect_to_order_form($pid) {

    // Get data form database
    $query = db_select('membership_products', 'p')
            ->condition('p.pid', $pid)
            ->fields('p', ['hash']);
    $product = $query->execute()->fetchAssoc();

    drupal_goto("megrendeles/", array('query' => array('pid' => $pid, 'hash' => $product['hash'])));
}

/**
 * Redirect to product order form.
 * @param type integer $pid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_redirect_to_period_order_form($pid) {

    // Get data form database
    $query = db_select('membership_period_products', 'p')
            ->condition('p.pid', $pid)
            ->fields('p', ['hash']);
    $product = $query->execute()->fetchAssoc();

    drupal_goto("period/megrendeles/", array('query' => array('pid' => $pid, 'hash' => $product['hash'])));
}

/**
 * Redirect to product order form.
 * @param type integer $pid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_redirect_period_product_list($query = array()) {
    return array('admin/config/membership/period/products',
        array(
            'query' => $query,
            'fragment' => '',
        ),
    );
}

/**
 * Provide default after order analytics code.
 * @return string.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_default_analytics_code() {
    $code = '<?php
// Sanitize the $_GET array.
foreach ($_GET as $key => $value) {
  $_GET[$key] = trim(check_plain($value));
}
?>

<script type="text/javascript">
ga("require", "ecommerce");

ga("ecommerce:addTransaction", {
  "id": \'<?php if (isset($_GET["transid"]))  print $_GET["transid"]; ?>\',                     // Transaction ID. Required
  "affiliation": \'<?php if (isset($_GET["store"]))    print $_GET["store"]; ?>\',   // Affiliation or store name
  "revenue": \'<?php if (isset($_GET["total"]))    print $_GET["total"]; ?>\',               // Grand Total
  "shipping": \'<?php if (isset($_GET["shipping"])) print $_GET["shipping"]; ?>\',                  // Shipping
  "tax": \'<?php if (isset($_GET["tax"]))      print $_GET["tax"]; ?>\'                     // Tax
});

ga("ecommerce:addItem", {
  "id": \'<?php if (isset($_GET["transid"]))  print $_GET["transid"]; ?>\',                     // Transaction ID. Required
  "name": \'<?php if (isset($_GET["name"])) print $_GET["name"]; ?>\',                // Product name. Required
  "sku": \'<?php if (isset($_GET["sku"]))      print $_GET["sku"]; ?>\',                    // SKU/code
  "category": \'<?php if (isset($_GET["category"])) print $_GET["category"]; ?>\',       // Category or variation
  "price": \'<?php if (isset($_GET["total"]))    print $_GET["total"]; ?>\',                 // Unit price
  "quantity": "1"                   // Quantity
});

ga("ecommerce:send");      // Send transaction and item data to Google Analytics.
</script>

<script>
  fbq("track", "Purchase", {
    value: <?php if (isset($_GET["total"])) print $_GET["total"]; ?>,
    currency: "HUF",
  });
</script>';

    return $code;
}

/**
 * Get memebership analytics block.
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_analytics_block() {
    $block = db_select('block_custom', 'b')
            ->fields('b')
            ->condition('b.info', 'membership_analytics')
            ->execute()
            ->fetchAssoc();
    if ($block) {
        return $block;
    }
    return FALSE;
}

/**
 * Get memebership analytics block.
 * @params type string $thank_you_link New thank you page link.
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_update_analytics_block_pages($thank_you_link) {
    $block = db_select('block', 'b')
            ->fields('b')
            ->condition('b.delta', 'membership_analytics')
            ->execute()
            ->fetchAssoc();
    if ($block) {
        $pages = strpos($block['pages'], $thank_you_link) === FALSE ? $block['pages'] . '
' . $thank_you_link : $block['pages'];
        db_update('block')
                ->fields(['pages' => $pages])
                ->condition('delta', 'membership_analytics', '=')
                ->execute();
        return TRUE;
    }
    return FALSE;
}
