<?php

/**
 * @file The global functions for the Barion API module.
 * @author Müller Péter <egyeves@gmail.com>
 */

/**
 * Check if a value is an integer.
 * @param mixed $variable The value.
 * @return boolean Returns TRUE if the value is an integer else returns FALSE.
 * @author Müller Péter <egyeves@gmail.com>
 */
function _barion_api_is_int($variable) {
    return ctype_digit(strval($variable));
}

/**
 * Check the email if correct or not.
 * @param string $email The email raw input.
 * @return boolean TRUE if the email looks correct.
 * @author Müller Péter <egyeves@gmail.com>
 */
function barion_api_is_valid_email($email) {
    $isValid = true;
    $atIndex = strrpos($email, "@");
    if (is_bool($atIndex) && !$atIndex) {
        $isValid = false;
    } else {
        $domain = substr($email, $atIndex + 1);
        $local = substr($email, 0, $atIndex);
        $localLen = strlen($local);
        $domainLen = strlen($domain);
        if ($localLen < 1 || $localLen > 64) {
            // Local part length exceeded.
            $isValid = false;
        } elseif ($domainLen < 1 || $domainLen > 255) {
            // Domain part length exceeded.
            $isValid = false;
        } elseif ($local[0] == '.' || $local[$localLen - 1] == '.') {
            // Local part starts or ends with '.'.
            $isValid = false;
        } elseif (preg_match('/\\.\\./', $local)) {
            // Local part has two consecutive dots.
            $isValid = false;
        } elseif (!preg_match('/^[A-Za-z0-9\\-\\.]+$/', $domain)) {
            // Character not valid in domain part.
            $isValid = false;
        } elseif (preg_match('/\\.\\./', $domain)) {
            // Domain part has two consecutive dots.
            $isValid = false;
        } elseif (!preg_match('/^(\\\\.|[A-Za-z0-9!#%&`_=\\/$\'*+?^{}|~.-])+$/', str_replace("\\\\", "", $local))) {
            // Character not valid in local part unless local part is quoted.
            if (!preg_match('/^"(\\\\"|[^"])+"$/', str_replace("\\\\", "", $local))) {
                $isValid = false;
            }
        }
        if ($isValid && !(checkdnsrr($domain, "MX") || checkdnsrr($domain, "A"))) {
            // Domain not found in DNS.
            $isValid = false;
        }
    }
    return $isValid;
}

/**
 * Check the email if correct or not in settings form.
 * @param string $email The email raw input.
 * @return boolean TRUE if the email looks correct.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_api_is_valid_email_inform($element, &$form_state) {
    $email = $element['#value'];
    $isValid = true;
    $atIndex = strrpos($email, "@");
    if (is_bool($atIndex) && !$atIndex) {
        $isValid = false;
    } else {
        $domain = substr($email, $atIndex + 1);
        $local = substr($email, 0, $atIndex);
        $localLen = strlen($local);
        $domainLen = strlen($domain);
        if ($localLen < 1 || $localLen > 64) {
            // Local part length exceeded.
            $isValid = false;
        } elseif ($domainLen < 1 || $domainLen > 255) {
            // Domain part length exceeded.
            $isValid = false;
        } elseif ($local[0] == '.' || $local[$localLen - 1] == '.') {
            // Local part starts or ends with '.'.
            $isValid = false;
        } elseif (preg_match('/\\.\\./', $local)) {
            // Local part has two consecutive dots.
            $isValid = false;
        } elseif (!preg_match('/^[A-Za-z0-9\\-\\.]+$/', $domain)) {
            // Character not valid in domain part.
            $isValid = false;
        } elseif (preg_match('/\\.\\./', $domain)) {
            // Domain part has two consecutive dots.
            $isValid = false;
        } elseif (!preg_match('/^(\\\\.|[A-Za-z0-9!#%&`_=\\/$\'*+?^{}|~.-])+$/', str_replace("\\\\", "", $local))) {
            // Character not valid in local part unless local part is quoted.
            if (!preg_match('/^"(\\\\"|[^"])+"$/', str_replace("\\\\", "", $local))) {
                $isValid = false;
            }
        }
        if ($isValid && !(checkdnsrr($domain, "MX") || checkdnsrr($domain, "A"))) {
            // Domain not found in DNS.
            $isValid = false;
        }
    }
    if (!$isValid) {
        form_error($element, t('The <em>!name</em> field must be valid email address.', array('!name' => $element['#title'])));
    }
//  return $isValid;
}

/**
 * Validates e-mail in the Drupal Form API fields.
 * @param type $element
 * @param type $form_state
 * @author Müller Péter <egyeves@gmail.com>
 */
function barion_api_element_validate_email($element, &$form_state) {
    $value = trim($element['#value']);
    if (!barion_api_is_valid_email($value)) {
        form_error($element, t('%name must be an email.', array('%name' => $element['#title'])));
    }
}

/**
 * Creates a random alphanumeric string the given length.
 * @param integer $length Lenght of the string.
 * @param string $type The hash type.
 * @return string The random string.
 * @author Müller Péter <egyeves@gmail.com>
 */
function barion_api_make_hash($length = 32, $type = 'alfanumeric') {
    mt_srand((double) microtime() * 1000000);
    if ($type == 'alfanumeric') {
        $aZ09 = array_merge(range('A', 'Z'), range('a', 'z'), range(0, 9));
    } elseif ($type == 'alfa') {
        $aZ09 = array_merge(range('A', 'Z'), range('a', 'z'));
    } elseif ($type == 'bigalfa') {
        $aZ09 = array_merge(range('A', 'Z'));
    } elseif ($type == 'smallalfa') {
        $aZ09 = array_merge(range('a', 'z'));
    } else { // 'numeric'.
        $aZ09 = array_merge(range(0, 9));
    }
    $hash = '';
    for ($c = 0; $c < $length; $c++) {
        $hash .= $aZ09[mt_rand(0, count($aZ09) - 1)];
    }
    return $hash;
}

/**
 * If the IP is real it returns.
 * @return string The IP address if the IP is real else empty string.
 * @author Müller Péter <egyeves@gmail.com>
 */
function barion_api_get_real_ip() {
    $ip = $_SERVER['REMOTE_ADDR'];
    $ip = ip2long($ip);
    if ($ip !== false && $ip !== -1 && $ip !== 0) {
        $ip = long2ip($ip);
    } else {
        $ip = '';
    }
    return $ip;
}

/**
 * Easy look debug function.
 * @params type $var array.
 * @params type $json boolean flag.
 * @return string Formatted array tree.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_debug($var, $json = FALSE) {
    if ($json) {
        header('Content-Type: application/json');
        echo drupal_json_output($var);
    } else {
        echo '<pre>';
        print_r($var);
        echo '</pre>';
    }
}

/**
 * If the variable is defined, set new value.
 * @param type $form_state It has to contain [values] associative array
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_api_variable_set_from_form($form_state) {
    foreach ($form_state['values'] as $key => $value) {
        if (variable_get($key, 'no_var') !== 'no-var') {
            $value = barion_api_array_filter_empty_value($value, $form_state);
            variable_set($key, $value);
        }
    }
}

/**
 * Filter the array if the value is '' or null or false give back empty string;
 * @param type $form_state It has to contain [values] associative array
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_api_array_filter_empty_value($value, $form_state) {
    if (is_array($value) && isset($form_state['values']['array_filter'])) {
        $value = array_keys(array_filter($value));
        return $value;
    }
    return $value;
}

/**
 * Generate token.
 * @return string A hash code with 4-4-4 char and 5 number.
 * @author Müller Péter <egyeves@gmail.com>
 */
function barion_api_generate_token() {
    return barion_api_make_hash($length = 4, $type = 'bigalfa') . '-'
            . barion_api_make_hash($length = 4, $type = 'bigalfa') . '-'
            . barion_api_make_hash($length = 4, $type = 'bigalfa') . '-'
            . barion_api_make_hash($length = 5, $type = 'numeric');
}

/**
 * Created a hash code.
 * @return string A hash code with 2 char and 6 number.
 * @author Müller Péter <egyeves@gmail.com>
 */
function barion_api_get_hash() {
    return barion_api_make_hash($length = 2, $type = 'bigalfa') . barion_api_make_hash($length = 6, $type = 'numeric');
}

/**
 * Return the unique hash code.
 * @return string The unique has code.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function barion_api_get_transaction_unique_hash() {
    $hash = barion_api_get_hash();
    // Iterates until the hash code not unique.
    $unique = db_query_range('SELECT id FROM {barion_api_transactions} WHERE tr_hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    while (!$unique) {
        $hash = barion_api_get_hash();
        $unique = barion_api_is_transaction_hash_unique($hash);
    }
    return $hash;
}

/**
 * Check if the has code is unique.
 * @param string $hash The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function barion_api_is_transaction_hash_unique($hash) {
    $result = db_query_range('SELECT id FROM {barion_api_transactions} WHERE tr_hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    return (empty($result)) ? TRUE : FALSE;
}

/**
 * Return the unique hash code.
 * @return string The unique has code.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function barion_api_get_payment_unique_hash() {
    $hash = barion_api_get_hash();
    // Iterates until the hash code not unique.
    $unique = db_query_range('SELECT id FROM {barion_api_transactions} WHERE pr_hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    while (!$unique) {
        $hash = barion_api_get_hash();
        $unique = barion_api_is_payment_hash_unique($hash);
    }
    return $hash;
}

/**
 * Check if the has code is unique.
 * @param string $hash The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function barion_api_is_payment_hash_unique($hash) {
    $result = db_query_range('SELECT id FROM {barion_api_transactions} WHERE pr_hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    return (empty($result)) ? TRUE : FALSE;
}

/**
 * Price value formatter.
 * @param type number.
 * @return string Formatted.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_api_format_price($number) {
    return number_format($number, 0, ',', ' ') . ' Ft';
}

/**
 * Check if a string is serialized
 * @param string $string
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_api_is_serial($string) {
    return (@unserialize($string) !== false);
}

/**
 * Check if a string is serialized
 * @param string $bigger
 * @param string $smaller
 * @return absolute value number.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_api_get_abs_different($bigger, $smaller) {
    return abs(floatval($bigger) - floatval($smaller));
}

/**
 * Check if the uid is valid.
 * @param string $uid The hash code.
 * @return boolean FALSE if the uid is not valid.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_api_isset_uid($uid) {
    $result = db_query_range('SELECT count(uid) uid FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid))->fetchField();
    return (empty($result)) ? FALSE : TRUE;
}

/**
 * Adds a new role
 * @machine_name - the name of the role to be added
 * @weight - the optional display weight for role
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _barion_api_add_role($machine_name, $weight = 0) {
    $role = new stdClass();
    $role->name = $machine_name;
    $role->weight = $weight;
    user_role_save($role);
}

/**
 * Adds permissions to a role
 * @role_name - the name of the role to receive an update
 * @permissions - the array of permissions to add
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _barion_api_add_permissions($role_name, $permissions) {
    $role = user_role_load_by_name($role_name);
    user_role_grant_permissions($role->rid, $permissions);
}

/**
 * Return the redirect array to the list page.
 * @params type integer $oid.
 * @params type integer $payment_id.
 * @return array Redirection array.
 */
function barion_api_redirect_to_payment_in($oid, $payment_id) {
    // Get thank you page url
    $redirect = variable_get('barion_api_callback');

    drupal_goto($redirect . '/' . $oid . '/' . $payment_id);
}

/**
 * Return the redirect array to the list page.
 * @params type array $query.
 * @return array Redirection array.
 */
function barion_api_redirect_to_thank_you($query = array()) {
    // Get thank you page url
    $redirect = variable_get('barion_api_redirect');

    drupal_goto($redirect, array('query' => $query));
}

/**
 * Return all payment details.
 * @params type string $paymentid.
 * @return array Response data array.
 */
function barion_api_get_payment($paymentid) {

//    mdd($paymentid, true);

    watchdog('barion_api', '"%paymentid" azonosító lekérés.', array('%paymentid' => $paymentid, WATCHDOG_NOTICE));

    // Get response
    $response = db_select('barion_api_transactions', 't')->fields('t')->condition('payment', check_plain($paymentid))->execute()->fetchAssoc();

    // Unserialize the full response
    $response['response'] = unserialize($response['response']);

    watchdog('barion_api', '"%paymentid" response lekérés.', array('%paymentid' => json_encode($response), WATCHDOG_NOTICE));

    return $response;
}

/**
 * Return all Barion design elements.
 * @return array Contain all frontend element for the pay button.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_api_sources() {

    $sources = array(
        'barion-card-payment-mark-2017-200px' => 'assets/barion-smart-payment-banners/barion-card-payment-mark-2017-200px.png',
        'barion-card-payment-mark-2017-300px' => 'assets/barion-smart-payment-banners/barion-card-payment-mark-2017-300px.png',
        'barion-card-payment-mark-2017-400px' => 'assets/barion-smart-payment-banners/barion-card-payment-mark-2017-400px.png',
        'barion-card-payment-mark-2017-500px.jpg' => 'assets/barion-smart-payment-banners/barion-card-payment-mark-2017-500px.jpg.jpg',
        'barion-card-payment-mark-2017-500px' => 'assets/barion-smart-payment-banners/barion-card-payment-mark-2017-500px.png',
        'barion-card-payment-mark-2017-800px' => 'assets/barion-smart-payment-banners/barion-card-payment-mark-2017-800px.png',
        'barion-card-payment-mark-2017-1200px' => 'assets/barion-smart-payment-banners/barion-card-payment-mark-2017-1200px.png',
        'barion-blue-partner-600px' => 'assets/barion-smart-payment-banners/barion-blue-partner-600px.png',
        'barion-silver-partner-600px' => 'assets/barion-smart-payment-banners/barion-silver-partner-600px.png',
        'barion-gold-partner-600px' => 'assets/barion-smart-payment-banners/barion-gold-partner-600px.png',
        'barion-web-app' => 'assets/barion-smart-payment-banners/barion-web-app.png',
    );
    return $sources;
}

/**
 * Return all user tokens.
 * @params type array $uid All details of the order.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_api_get_tokens($uid = 0) {

    $date = date('Ym');
    $tokens = array();

    // Get all tokens of the user
    $tokens = db_select('barion_api_tokens', 't')
            ->fields('t')
            ->condition('t.uid', $uid)
            ->where('CONCAT(t.card_year, t.card_month) >= :date', array(':date' => $date))
            ->orderBy('t.created')
            ->execute()
            ->fetchAll();

    return $tokens;
}

/**
 * Return token string.
 * @params type string $id Token id.
 * @return type string Token string.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function barion_api_get_token($id) {

    // Get all tokens of the user
    $token = db_select('barion_api_tokens', 't')
            ->fields('t', array('token'))
            ->condition('t.id', $id)
            ->execute()
            ->fetchField();

    return $token;
}
