<?php

/**
 * @file
 * Az Email Marketer rutinjai
 * 
 * Azokat a függvényket tartalmazza, amik összekötik az Email Marketert
 * a függyvényen keresztül a Drupal rendszereivel.  
 *  
 * @see http://www.interspire.com/emailmarketer/pdf/XMLApiDocumentation.pdf
 * @see https://www.interspire.com/support/kb/questions/1217/Email+Marketer+XML+API+usage+and+examples
 * 
 */

/**
 * Kicsréli a & " ' jeleket a html megfelelőjükre, hogy az XML értelmezővel
 * ne legyen probléma.
 * 
 * @param string $text
 *   Szöveg, amiben ki kell cserélni a felsorolt jeleket.
 * 
 * @return string
 *   A szöveg amiben a felsorolt jelek a html megfelelőjükkél lettek alakítva.
 */
function _entitles_swap ($text) { 
  $character = array ("&",'"', "'");
  $name = array ("&amp;","&quot;", "&apos;");
  $result = str_replace($character, $name, $text);
  return $result;
}

/**
 * Létrehozza teljes XML fájlt, hogy elküldhető legyen a Email Marketer
 * XML api-jának úgy, hogy hozzáadja a bejelentkezéshez szükséges előtagokat.
 * 
 * @param string $xml
 *   Az XML kódja az adott funkciónak.
 *
 * @return string
 *   Az XML kód a bejelentkezési információkkal kiegésztve.
 */
function _em_create_xml($xml){
  $XML_username = variable_get('membership_em_xml_username');
  $XML_token = variable_get('membership_em_xml_token');

  $new_xml ="<xmlrequest>
    <username>$XML_username</username>
    <usertoken>$XML_token</usertoken>";
  $new_xml .= $xml;
  $new_xml .= '</xmlrequest>';
  $new_xml = _entitles_swap($new_xml);
  return $new_xml;
}

/**
 * Elküldi a kész XML fájlt az Email Marketer XML api-jának. 
 * 
 * @param string $xml
 *   A teljesen összállított XML, amit a rendszer elküld az XML api-nak.
 * 
 * @return
 *   Egy objektum osztályt ad vissza (SimpleXMLElement) az XML dolumentummal.
 */
function _em_send_xml($xml) {
  $XML_path = variable_get('membership_em_xml_path');
  $ch = curl_init($XML_path);
  curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
  curl_setopt($ch, CURLOPT_POST, 1);
  curl_setopt($ch, CURLOPT_POSTFIELDS, $xml);
  $result = @curl_exec($ch);
  if($result === FALSE) {
    drupal_set_message(t('Hiba történt. Az email küldő rendszerrel nem lehet felvenni a kapcsolatot. Kérjük értestíse az oldal adminisztrátorát!'), 'error');
    watchdog('membership', t('Nem lehet elérni az hírelvél küldő rendszert! A hiba helye: email_marketer.api.php Függvény: _em_send_xml.'), array(), WATCHDOG_ERROR);
  } else {
    $xml_doc = simplexml_load_string($result);
    
    // drupal_set_message( 'status is '. $xml_doc->status);
    // if ($xml_doc->status == 'SUCCESS') {
      // drupal_set_message(print_r($result));
    // } else {
      // drupal_set_message('Error is '. $xml_doc->errormessage);
    // }
    return $xml_doc;
  }
}

/**
 * Leteszteli, hogy működik-e az XML API
 * 
 * @return string
 *   Egy objektum osztályt ad vissza (SimpleXMLElement) az XML dolumentummal.
 * 
 * @code
 * em_xml_api_test();
 * @endcode
 * 
 */
function em_xml_api_test() {
  $xml ='<requesttype>authentication</requesttype>
    <requestmethod>xmlapitest</requestmethod>
    <details>
    </details>
  ';
  $xml= _em_create_xml($xml);
  $result = _em_send_xml($xml);
  return $result;
}

/**
 * Egy felhasználót ad hozzá a megadott hírlevél listához.
 * 
 * @param array $subscriber
 *   Egy megfelelő formátumben összeállított tömb.
 * 
 * @return string
 *   Egy objektum osztályt ad vissza (SimpleXMLElement) az XML dolumentummal.
 * 
 * @code
 * $subscriber = array(
 *   'emailaddress' => 'email@domain.com',
 *   'mailinglist'  => 1,
 *   'format'       => 'text',  // html vagy text
 *   'confirmed'    => 'yes',   // (yes, 1, true) vagy (no, 0, false)
 *   'customfields' => array(   // items
 *     array('fieldid' => 1,'value'   => 'John Smith'),
 *     array('fieldid' => 2,'value'   => 'male'),
 *   ),
 * );
 * em_add_subscriber_to_list($subscriber);
 * @endcode
 * 
 */
function em_add_subscriber_to_list($subscriber = array()) {
  $xml = '<requesttype>subscribers</requesttype>
    <requestmethod>AddSubscriberToList</requestmethod>
    <details>
      <emailaddress>'.$subscriber['emailaddress'].'</emailaddress>
      <mailinglist>' .$subscriber['mailinglist'] .'</mailinglist>
      <format>'      .$subscriber['format']      .'</format>
      <confirmed>'   .$subscriber['confirmed']   .'</confirmed>
      <customfields>';
  foreach ($subscriber['customfields'] as $customfield) {
    $xml .= PHP_EOL . '<item>
      <fieldid>'.$customfield['fieldid'].'</fieldid>
      <value>'.$customfield['value'].'</value>
    </item>';
  }
        
  $xml .= PHP_EOL.'</customfields>
    </details>
  ';
  $xml = _em_create_xml($xml);
  $result = _em_send_xml($xml);
  return $result;
}

/**
 * Töröl egy feliratkozót a megadott hírlevél listáról
 * 
 * @param string $email
 *   A törlésre kijelölt felhasználó e-mail címe.
 * 
 * @param int $listid
 *   A hírlevél lista sorszáma, amiről a felhasználót törölni kell.
 * 
 * @return string
 *   Egy objektum osztályt ad vissza (SimpleXMLElement) az XML dolumentummal.
 *
 * @code
 * em_delete_subscriber('email@domain.com',1);
 * @endcode 
 */
function em_delete_subscriber($email, $listid) {
  $xml = "<requesttype>subscribers</requesttype>
    <requestmethod>DeleteSubscriber</requestmethod>
    <details>
      <emailaddress>$email</emailaddress>
      <list>$listid</list>
    </details>
  ";
  $xml = _em_create_xml($xml);
  $result = _em_send_xml($xml);
  return $result;
}

/**
 * Beolvassa a megadott hírlevél listák egyedi mezőit.
 * 
 * @param int
 *   Felsorolásban meg kell adni a hírlevél listák, azonosítóit.
 *   Leglább 1 paramétert vár.
 * 
 * @return string
 *   Egy objektum osztályt ad vissza (SimpleXMLElement) az XML dolumentummal.
 * 
 * @code
 * em_get_custom_fields(1,2,4);
 * @endcode
 * 
 */
function em_get_custom_fields() {
  $arg_list = func_get_args();
  $lists = '';
  foreach ($arg_list as $listid) {
    $lists .= "<listids>$listid</listids>".PHP_EOL;
  }
  $xml ="<requesttype>lists</requesttype>
    <requestmethod>GetCustomFields</requestmethod>
    <details>
    $lists
    </details>
  ";
  $xml = _em_create_xml($xml);
  $result = _em_send_xml($xml);
  return $result;
}

/**
 * Beolvassa az összes hírlevél listát
 * 
 * @return string
 *   Egy objektum osztályt ad vissza (SimpleXMLElement) az XML dolumentummal.
 * 
 * @code
 * em_get_lists();
 * @endcode
 * 
 */
function em_get_lists() {
    $xml ='<requesttype>user</requesttype>
    <requestmethod>GetLists</requestmethod>
    <details>
    </details>
  ';
  $xml= _em_create_xml($xml);
  $result = _em_send_xml($xml);
  return $result;
}

/**
 * Megszámolja és kilistázza az adott listán lévő feliratkozókat, akiknek az
 * e-mail címük megfelel az adott mintának.
 * 
 * @param string $email
 *   A keresésre kijelölt email cím vagy annak részlete. Üresen hagyva minden
 *   email címet ki fog listázni.
 * 
 * @param int $listid
 *   A hírlevél lista sorszáma, amin kersni kell.
 *  
 * @return string
 *   Egy objektum osztályt ad vissza (SimpleXMLElement) az XML dolumentummal.
 * 
 * @code
 * em_get_subscribers('syrah', 1);
 * em_get_subscribers('', 1);
 * @endcode
 * 
 */
function em_get_subscribers($email, $listid) {
  $xml = "<requesttype>subscribers</requesttype>
    <requestmethod>GetSubscribers</requestmethod>
    <details>
      <searchinfo>
        <List>$listid</List>
        <Email>$email</Email>
      </searchinfo>
    </details>
  ";
  $xml = _em_create_xml($xml);
  $result = _em_send_xml($xml);
  return $result;
}
 
/**
 * Leellenőrzi, hogy a megedott feliratkozó rajta van-e a megadott listán és
 * visszaadja a sorszámát.
 * 
 * @param string $email
 *   A keresendő e-mail cím.
 * 
 * @param int $listid
 *   A hírlevél lista sorszáma, amin az email címet keresni kell.
 * 
 * @return string
 *   Egy objektum osztályt ad vissza (SimpleXMLElement) az XML dolumentummal.
 *
 * @code
 * em_is_subscriber_on_list('email@domain.com', 1)
 * @endcode 
 */
function em_is_subscriber_on_list($email, $listid) {
  $xml = "<requesttype>subscribers</requesttype>
    <requestmethod>IsSubscriberOnList</requestmethod>
    <details>
      <Email>$email</Email>  
      <List>$listid</List>
    </details>
  ";
  $xml = _em_create_xml($xml);
  $result = _em_send_xml($xml);
  return $result;
}
