<?php

/**
 * @file The global functions for the Certify module.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */

/**
 * Check if a value is an integer.
 * @param mixed $variable The value.
 * @return boolean Returns TRUE if the value is an integer else returns FALSE.
 * @author Müller Péter <egyeves@gmail.com>
 */
function _membership_is_int($variable) {
    return ctype_digit(strval($variable));
}

/**
 * Get the fields form the certify_candidate_groups table.
 * @return string The name field value.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_get_user_fields($uid) {
    return db_query_range('SELECT uid, name, mail FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid,))->fetchAssoc();
}

/**
 * Easy look debug function.
 * @params type $var array.
 * @params type $json boolean flag.
 * @return string Formatted array tree.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function mdd($var, $json = FALSE) {
    if ($json) {
        header('Content-Type: application/json');
        echo drupal_json_output($var);
        exit();
    } else {
        echo '<pre>';
        print_r($var);
        echo '</pre>';
    }
}

/**
 * Filter the array if the value is '' or null or false give back empty string;
 * @param type $form_state It has to contain [values] associative array
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_array_filter_empty_value($value, $form_state) {
    if (is_array($value) && isset($form_state['values']['array_filter'])) {
        $value = array_keys(array_filter($value));
        return $value;
    }
    return $value;
}

/**
 * Price value formatter.
 * @param type number.
 * @return string Formatted.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_format_price($number) {
    return number_format($number, 0, ',', ' ') . ' ' . variable_get('membership_firm_invoice_currency', 'Ft');
}

/**
 * Check if a string is serialized
 * @param string $string
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_is_serial($string) {
    return (@unserialize($string) !== false);
}

/**
 * Check if a string is serialized
 * @param string $bigger
 * @param string $smaller
 * @return absolute value number.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_get_abs_different($bigger, $smaller) {
    return abs(floatval($bigger) - floatval($smaller));
}

/**
 * Check if the uid is valid.
 * @param string $uid The hash code.
 * @return boolean FALSE if the uid is not valid.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_isset_uid($uid) {
    $result = db_query_range('SELECT count(uid) uid FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid))->fetchField();
    return (empty($result)) ? FALSE : TRUE;
}

/**
 * Adds a new role
 * @machine_name - the name of the role to be added
 * @weight - the optional display weight for role
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_add_role($machine_name, $weight = 0) {
    $role = new stdClass();
    $role->name = $machine_name;
    $role->weight = $weight;
    user_role_save($role);
}

/**
 * Adds permissions to a role
 * @role_name - the name of the role to receive an update
 * @permissions - the array of permissions to add
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_add_permissions($role_name, $permissions) {
    $role = user_role_load_by_name($role_name);
    user_role_grant_permissions($role->rid, $permissions);
}

/**
 * Return the two digits string
 * @params type integer.
 * @return string.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_two_digits($int) {
    if (strlen($int) == 1) {
        return '0' . $int;
    } else {
        return $int;
    }
}

/**
 * Show full name in set order.
 * @params type string $lastname.
 * @params type string $firstname.
 * @return string Fullname.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _membership_show_name($lastname, $firstname) {
    if (variable_get('membership_basic_name', '1') == 1) {
        return _capitalize($firstname) . ' ' . _capitalize($lastname);
    } else {
        return _capitalize($lastname) . ' ' . _capitalize($firstname);
    }
}

/**
 * Egy e-mailt küld a rendszerből a felhasználónak.
 *
 * @param string $name
 * A címzett neve.
 *
 * @param type string $email.
 * A címzett e-mail címe.
 *
 * @param type string $subject.
 * Az e-mail tárgya.
 *
 * @param string $message
 * Az e-mail üzetnet.
 *
 * @author Müller Péter <egyeves@gmail.com>
 */
function _membership_email_send($name, $email, $subject, $message) {

    $settings_mail_from_name = variable_get('certify_email_sender_name');
    $settings_mail_from_email = variable_get('certify_email_sender_email');

    $to = '"' . '=?UTF-8?B?' . base64_encode($name) . '?=' . '" <' . $email . '>';

    $header = 'MIME-Version: 1.0' . PHP_EOL;
    $header .= 'Content-type: text/plain; format=flowed; charset="UTF-8"' . PHP_EOL;
    $header .= 'Content-Transfer-Encoding: 8bit' . PHP_EOL;
    $header .= 'From: "' . '=?UTF-8?B?' . base64_encode($settings_mail_from_name) . '?=' . '" <' . $settings_mail_from_email . '>' . PHP_EOL;
    $header .= 'Reply-To: ' . $settings_mail_from_email . PHP_EOL;
    $header .= 'Return-Path: ' . $settings_mail_from_email . PHP_EOL;
    $header .= 'X-Mailer: PHP-' . phpversion() . PHP_EOL;

    mail($to, '=?UTF-8?B?' . base64_encode($subject) . '?=', $message, $header);
}

/**
 * Get all of order details.
 * @param type integer $oid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_order_get_details($oid) {

    // Get data form database
    $query = db_select('membership_orders', 'o')
            ->condition('o.oid', $oid)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid']);
    $order = $query->execute()->fetchAssoc();

    return $order;
}

/**
 * Get all of order details.
 * @param type integer $oid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_upsell_order_get_details($oid) {

    // Get data form database
    $query = db_select('membership_orders', 'o')
            ->condition('o.uoid', $oid)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid']);
    $order = $query->execute()->fetchAssoc();

    return $order;
}

/**
 * Get all of order details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_order_get_all_details_object($oid) {

    // Get data form database
    $query = db_select('membership_orders', 'o')
            ->condition('o.oid', $oid)
            ->fields('o');
    $query->leftJoin('membership_users', 'u', '(o.id = u.id)');
    $query->fields('u');
    $query->leftJoin('membership_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid', 'commission', 'vat']);
    $order = $query->execute()->fetchObject();

    return $order;
}

/**
 * Get user's last order.
 * @param type integer $uid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_user_last_order_object($uid) {

    // Get data form database
    $query = db_select('membership_orders', 'o')
            ->condition('o.id', $uid)
            ->fields('o')
            ->orderBy('created', 'DESC');
    $query->leftJoin('membership_products', 'p', '(o.pid = p.pid)');
    $query->fields('p', ['hash', 'name', 'proforma_name', 'pgid', 'commission', 'vat']);
    $query->range(0, 1);
    $order = $query->execute()->fetchObject();

    return $order;
}

/**
 * Get order details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_order_get_details_object($oid) {

    // Get data form database
    $query = db_select('membership_orders', 'o')
            ->condition('o.oid', $oid)
            ->fields('o');
    $order = $query->execute()->fetchObject();

    return $order;
}

/**
 * Get product details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_product_get_details_object($pid) {

    // Get data form database
    $query = db_select('membership_products', 'p')
            ->condition('p.pid', $pid)
            ->fields('p');
    $product = $query->execute()->fetchObject();

    return $product;
}

/**
 * Get product details object.
 * @param type integer $oid
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_product_get_upsell_details_object($pid) {

    // Get data form database
    $query = db_select('membership_upsell_products', 'u')
            ->condition('u.pid', $pid)
            ->fields('u', array('upid', 'status', 'price', 'title', 'description'));
    $query->leftJoin('membership_products', 'p', '(u.upid = p.pid)');
    $query->fields('p');
    $product = $query->execute()->fetchObject();

    return $product;
}

/**
 * Get all of membership user details object.
 * @param type integer $id The user id column in the orders table.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_user_details_object($id) {

    // Get data form database
    // $user = db_query_range('SELECT * FROM {membership_users} WHERE id = :id', 0, 1, array(':id' => $order->id,))->fetchObject();

    $query = db_select('membership_users', 'u')
            ->condition('u.id', $id)
            ->fields('u');
    $user = $query->execute()->fetchObject();

    return $user;
}

/**
 * Get all of membership user details object.
 * @param type integer $uid The user id column in the orders table.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_user_details_object_by_uid($uid) {

    // Get data form database
    // $user = db_query_range('SELECT * FROM {membership_users} WHERE id = :id', 0, 1, array(':id' => $order->id,))->fetchObject();

    $query = db_select('membership_users', 'u')
            ->condition('u.uid', $uid)
            ->fields('u');
    $user = $query->execute()->fetchObject();

    return $user;
}

/**
 * Get all of membership user details object.
 * @param type integer $uid The user id column in the orders table.
 * @return object.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_user_details_object_by_email($email) {

    // Get data form database
    // $user = db_query_range('SELECT * FROM {membership_users} WHERE id = :id', 0, 1, array(':id' => $order->id,))->fetchObject();

    $query = db_select('membership_users', 'u')
            ->condition('u.email', check_plain(trim($email)))
            ->fields('u');
    $user = $query->execute()->fetchObject();

    return $user;
}

/**
 * Get all of order details.
 * @param type integer $oid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_user_get_cards($uid) {

    if (module_exists('barion_api')) {
        // Get data form database
        $query = db_select('barion_api_tokens', 'c')
                ->condition('c.uid', $uid)
                ->condition('c.created', '', '!=')
                ->fields('c');
        $tokens = $query->execute()->fetchAll();

        $cards = [null => t('Új bankkártya')];
        foreach ($tokens as $token) {
            $expire = date($token->card_year . '-' . _membership_two_digits((int) $token->card_month) . '-01');
            if ($expire > date('Y-m-d')) {
                $cards[$token->id] = $token->card_type . ' xx' . $token->card_mask . ' (' . _membership_two_digits((int) $token->card_month) . '/' . substr($token->card_year, 2, 2) . ')';
            }
        }

        return $cards;
    }
    return false;
}

/**
 * Get all products list.
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_products_list() {

    // Get data form database
    $query = db_select('membership_products', 'p')
            ->fields('p');
    $products = $query->execute()->fetchAll();

    foreach ($products as $product) {
        $list[$product->pid] = $product->name . ' (' . _membership_format_price($product->price) . ')';
    }

    return $list;
}

/**
 * Redirect thank you page with analytics params.
 * @param type integer $oid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_redirect_to_thank_you($oid, $thank_you_link = '') {

    if ($thank_you_link == '') {
        $thank_you_link = variable_get('membership_thankyou_link');
    }

    // Get order details
    $order = membership_order_get_details($oid);

    // Átiránytás beállítása, hogy a köszönöm oldalon a Google Analytics tudja rögzíteni az adatokat.
    $country = db_query('SELECT name FROM {membership_country} WHERE code = :code', array(':code' => $order['country']))->fetchField();
    $product_name = $order['name'];
    $product_group_name = db_query('SELECT name FROM {membership_product_groups} WHERE pgid = :pgid', array(':pgid' => $order['pgid']))->fetchField();

    $query = [
        'transid' => $oid,
        'store' => variable_get('membership_site_name'),
        'total' => $order['price'],
        'tax' => $order['price'] * $order['vat'] / 100,
        'shipping' => $order['shipping'],
        'city' => $order['city'],
        'state' => '',
        'country' => $country,
        'sku' => $order['pid'],
        'name' => $product_name,
        'category' => $product_group_name,
        'email' => $order['email'],
        'muid' => $order['id'],
    ];
    drupal_goto($thank_you_link, array('query' => $query));
}

/**
 * Redirect to product order form.
 * @param type integer $pid
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_redirect_to_order_form($pid) {

    // Get data form database
    $query = db_select('membership_products', 'p')
            ->condition('p.pid', $pid)
            ->fields('p', ['hash']);
    $product = $query->execute()->fetchAssoc();

    drupal_goto("megrendeles/", array('query' => array('pid' => $pid, 'hash' => $product['hash'])));
}

/**
 * Provide default after order analytics code.
 * @return string.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_default_analytics_code() {
    $code = '<?php
// Sanitize the $_GET array.
foreach ($_GET as $key => $value) {
  $_GET[$key] = trim(check_plain($value));
}
?>

<script type="text/javascript">
ga("require", "ecommerce");

ga("ecommerce:addTransaction", {
  "id": \'<?php if (isset($_GET["transid"]))  print $_GET["transid"]; ?>\',                     // Transaction ID. Required
  "affiliation": \'<?php if (isset($_GET["store"]))    print $_GET["store"]; ?>\',   // Affiliation or store name
  "revenue": \'<?php if (isset($_GET["total"]))    print $_GET["total"]; ?>\',               // Grand Total
  "shipping": \'<?php if (isset($_GET["shipping"])) print $_GET["shipping"]; ?>\',                  // Shipping
  "tax": \'<?php if (isset($_GET["tax"]))      print $_GET["tax"]; ?>\'                     // Tax
});

ga("ecommerce:addItem", {
  "id": \'<?php if (isset($_GET["transid"]))  print $_GET["transid"]; ?>\',                     // Transaction ID. Required
  "name": \'<?php if (isset($_GET["name"])) print $_GET["name"]; ?>\',                // Product name. Required
  "sku": \'<?php if (isset($_GET["sku"]))      print $_GET["sku"]; ?>\',                    // SKU/code
  "category": \'<?php if (isset($_GET["category"])) print $_GET["category"]; ?>\',       // Category or variation
  "price": \'<?php if (isset($_GET["total"]))    print $_GET["total"]; ?>\',                 // Unit price
  "quantity": "1"                   // Quantity
});

ga("ecommerce:send");      // Send transaction and item data to Google Analytics.
</script>

<script>
  fbq("track", "Purchase", {
    value: <?php if (isset($_GET["total"])) print $_GET["total"]; ?>,
    currency: "HUF",
  });
</script>';

    return $code;
}

/**
 * Get memebership analytics block.
 * @return array.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_get_analytics_block() {
    $block = db_select('block_custom', 'b')
            ->fields('b')
            ->condition('b.info', 'membership_analytics')
            ->execute()
            ->fetchAssoc();
    if ($block) {
        return $block;
    }
    return FALSE;
}

/**
 * Get memebership analytics block.
 * @params type string $thank_you_link New thank you page link.
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function membership_update_analytics_block_pages($thank_you_link) {
    $block = db_select('block', 'b')
            ->fields('b')
            ->condition('b.delta', 'membership_analytics')
            ->execute()
            ->fetchAssoc();
    if ($block) {
        $pages = strpos($block['pages'], $thank_you_link) === FALSE ? $block['pages'] . '
' . $thank_you_link : $block['pages'];
        db_update('block')
                ->fields(['pages' => $pages])
                ->condition('delta', 'membership_analytics', '=')
                ->execute();
        return TRUE;
    }
    return FALSE;
}
