<?php

/**
 * @file The global functions for the Dreams Dashboard module.
 * @author Müller Péter <egyeves@gmail.com>
 */

/**
 * Check if a value is an integer.
 * @param mixed $variable The value.
 * @return boolean Returns TRUE if the value is an integer else returns FALSE.
 * @author Müller Péter <egyeves@gmail.com>
 */
function _dreams_is_int($variable) {
    return ctype_digit(strval($variable));
}

/**
 * Check if a value is a float.
 * @param mixed $variable The value.
 * @return boolean Returns TRUE if the value is an integer else returns FALSE.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _dreams_is_float($variable) {
    $variable = str_replace(',', '.', $variable);
    $float = is_float((float) $variable);
    return (($float && (string) ((float) $variable) == $variable) || ctype_digit(strval($variable)));
}

/**
 * Check the email if correct or not.
 * @param string $email The email raw input.
 * @return boolean TRUE if the email looks correct.
 * @author Müller Péter <egyeves@gmail.com>
 */
function _dreams_is_valid_email($email) {
    $isValid = true;
    $atIndex = strrpos($email, "@");
    if (is_bool($atIndex) && !$atIndex) {
        $isValid = false;
    } else {
        $domain = substr($email, $atIndex + 1);
        $local = substr($email, 0, $atIndex);
        $localLen = strlen($local);
        $domainLen = strlen($domain);
        if ($localLen < 1 || $localLen > 64) {
            // Local part length exceeded.
            $isValid = false;
        } elseif ($domainLen < 1 || $domainLen > 255) {
            // Domain part length exceeded.
            $isValid = false;
        } elseif ($local[0] == '.' || $local[$localLen - 1] == '.') {
            // Local part starts or ends with '.'.
            $isValid = false;
        } elseif (preg_match('/\\.\\./', $local)) {
            // Local part has two consecutive dots.
            $isValid = false;
        } elseif (!preg_match('/^[A-Za-z0-9\\-\\.]+$/', $domain)) {
            // Character not valid in domain part.
            $isValid = false;
        } elseif (preg_match('/\\.\\./', $domain)) {
            // Domain part has two consecutive dots.
            $isValid = false;
        } elseif (!preg_match('/^(\\\\.|[A-Za-z0-9!#%&`_=\\/$\'*+?^{}|~.-])+$/', str_replace("\\\\", "", $local))) {
            // Character not valid in local part unless local part is quoted.
            if (!preg_match('/^"(\\\\"|[^"])+"$/', str_replace("\\\\", "", $local))) {
                $isValid = false;
            }
        }
        if ($isValid && !(checkdnsrr($domain, "MX") || checkdnsrr($domain, "A"))) {
            // Domain not found in DNS.
            $isValid = false;
        }
    }
    return $isValid;
}

/**
 * Check the email if correct or not in settings form.
 * @param string $email The email raw input.
 * @return boolean TRUE if the email looks correct.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _dreams_is_valid_email_inform($element, &$form_state) {
    $email = $element['#value'];
    $isValid = true;
    $atIndex = strrpos($email, "@");
    if (is_bool($atIndex) && !$atIndex) {
        $isValid = false;
    } else {
        $domain = substr($email, $atIndex + 1);
        $local = substr($email, 0, $atIndex);
        $localLen = strlen($local);
        $domainLen = strlen($domain);
        if ($localLen < 1 || $localLen > 64) {
            // Local part length exceeded.
            $isValid = false;
        } elseif ($domainLen < 1 || $domainLen > 255) {
            // Domain part length exceeded.
            $isValid = false;
        } elseif ($local[0] == '.' || $local[$localLen - 1] == '.') {
            // Local part starts or ends with '.'.
            $isValid = false;
        } elseif (preg_match('/\\.\\./', $local)) {
            // Local part has two consecutive dots.
            $isValid = false;
        } elseif (!preg_match('/^[A-Za-z0-9\\-\\.]+$/', $domain)) {
            // Character not valid in domain part.
            $isValid = false;
        } elseif (preg_match('/\\.\\./', $domain)) {
            // Domain part has two consecutive dots.
            $isValid = false;
        } elseif (!preg_match('/^(\\\\.|[A-Za-z0-9!#%&`_=\\/$\'*+?^{}|~.-])+$/', str_replace("\\\\", "", $local))) {
            // Character not valid in local part unless local part is quoted.
            if (!preg_match('/^"(\\\\"|[^"])+"$/', str_replace("\\\\", "", $local))) {
                $isValid = false;
            }
        }
        if ($isValid && !(checkdnsrr($domain, "MX") || checkdnsrr($domain, "A"))) {
            // Domain not found in DNS.
            $isValid = false;
        }
    }
    if (!$isValid) {
        form_error($element, t('The <em>!name</em> field must be valid email address.', array('!name' => $element['#title'])));
    }
//  return $isValid;
}

/**
 * Validates e-mail in the Drupal Form API fields.
 * @param type $element
 * @param type $form_state
 * @author Müller Péter <egyeves@gmail.com>
 */
function _dreams_element_validate_email($element, &$form_state) {
    $value = trim($element['#value']);
    if (!dreams_is_valid_email($value)) {
        form_error($element, t('%name must be an email.', array('%name' => $element['#title'])));
    }
}

/**
 * Creates a random alphanumeric string the given length.
 * @param integer $length Lenght of the string.
 * @param string $type The hash type.
 * @return string The random string.
 * @author Müller Péter <egyeves@gmail.com>
 */
function dreams_make_hash($length = 32, $type = 'alfanumeric') {
    mt_srand((double) microtime() * 1000000);
    if ($type == 'alfanumeric') {
        $aZ09 = array_merge(range('A', 'Z'), range('a', 'z'), range(0, 9));
    } elseif ($type == 'alfa') {
        $aZ09 = array_merge(range('A', 'Z'), range('a', 'z'));
    } elseif ($type == 'bigalfa') {
        $aZ09 = array_merge(range('A', 'Z'));
    } elseif ($type == 'smallalfa') {
        $aZ09 = array_merge(range('a', 'z'));
    } else { // 'numeric'.
        $aZ09 = array_merge(range(0, 9));
    }
    $hash = '';
    for ($c = 0; $c < $length; $c++) {
        $hash .= $aZ09[mt_rand(0, count($aZ09) - 1)];
    }
    return $hash;
}

/**
 * If the IP is real it returns.
 * @return string The IP address if the IP is real else empty string.
 * @author Müller Péter <egyeves@gmail.com>
 */
function _dreams_get_real_ip() {
    $ip = $_SERVER['REMOTE_ADDR'];
    $ip = ip2long($ip);
    if ($ip !== false && $ip !== -1 && $ip !== 0) {
        $ip = long2ip($ip);
    } else {
        $ip = '';
    }
    return $ip;
}

/**
 * Easy look debug function.
 * @params type $var array.
 * @params type $json boolean flag.
 * @return string Formatted array tree.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function ddd($var, $json = FALSE) {
    if ($json) {
        header('Content-Type: application/json');
        echo drupal_json_output($var);
        exit();
    } else {
        echo '<pre>';
        print_r($var);
        echo '</pre>';
    }
}

/**
 * If the variable is defined, set new value.
 * @param type $form_state It has to contain [values] associative array
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function dreams_variable_set_from_form($form_state) {
    foreach ($form_state['values'] as $key => $value) {
        if (variable_get($key, 'no_var') !== 'no-var') {
            $value = dreams_array_filter_empty_value($value, $form_state);
            variable_set($key, $value);
        }
    }
}

/**
 * Filter the array if the value is '' or null or false give back empty string;
 * @param type $form_state It has to contain [values] associative array
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _dreams_array_filter_empty_value($value, $form_state) {
    if (is_array($value) && isset($form_state['values']['array_filter'])) {
        $value = array_keys(array_filter($value));
        return $value;
    }
    return $value;
}

/**
 * Created a hash code.
 * @return string A hash code with 2 char and 6 number.
 * @author Müller Péter <egyeves@gmail.com>
 */
function dreams_get_hash() {
    return dreams_make_hash($length = 2, $type = 'bigalfa') . dreams_make_hash($length = 6, $type = 'numeric');
}

/**
 * Return the unique hash code.
 * @return string The unique has code.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_get_transaction_unique_hash() {
    $hash = dreams_get_hash();
    // Iterates until the hash code not unique.
    $unique = db_query_range('SELECT id FROM {dreams_transactions} WHERE tr_hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    while (!$unique) {
        $hash = dreams_get_hash();
        $unique = dreams_is_transaction_hash_unique($hash);
    }
    return $hash;
}

/**
 * Check if the has code is unique.
 * @param string $hash The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_is_transaction_hash_unique($hash) {
    $result = db_query_range('SELECT id FROM {dreams_transactions} WHERE tr_hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    return (empty($result)) ? TRUE : FALSE;
}

/**
 * Return the unique hash code.
 * @return string The unique has code.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_get_payment_unique_hash() {
    $hash = dreams_get_hash();
    // Iterates until the hash code not unique.
    $unique = db_query_range('SELECT id FROM {dreams_transactions} WHERE pr_hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    while (!$unique) {
        $hash = dreams_get_hash();
        $unique = dreams_is_payment_hash_unique($hash);
    }
    return $hash;
}

/**
 * Check if the has code is unique.
 * @param string $hash The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_is_payment_hash_unique($hash) {
    $result = db_query_range('SELECT id FROM {dreams_transactions} WHERE pr_hash = :hash', 0, 1, array(':hash' => $hash))->fetchField();
    return (empty($result)) ? TRUE : FALSE;
}

/**
 * Price value formatter.
 * @param type number.
 * @return string Formatted.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _dreams_format_price($number) {
    return number_format($number, 0, ',', ' ') . ' Ft';
}

/**
 * Check if a string is serialized
 * @param string $string
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _dreams_is_serial($string) {
    return (@unserialize($string) !== false);
}

/**
 * Check if a string is serialized
 * @param string $bigger
 * @param string $smaller
 * @return absolute value number.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _dreams_get_abs_different($bigger, $smaller) {
    return abs(floatval($bigger) - floatval($smaller));
}

/**
 * Check if the uid is valid.
 * @param string $uid The hash code.
 * @return boolean FALSE if the uid is not valid.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _dreams_isset_uid($uid) {
    $result = db_query_range('SELECT count(uid) uid FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid))->fetchField();
    return (empty($result)) ? FALSE : TRUE;
}

/**
 * Adds a new role
 * @machine_name - the name of the role to be added
 * @weight - the optional display weight for role
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _dreams_add_role($machine_name, $weight = 0) {
    $role = new stdClass();
    $role->name = $machine_name;
    $role->weight = $weight;
    user_role_save($role);
}

/**
 * Adds permissions to a role
 * @role_name - the name of the role to receive an update
 * @permissions - the array of permissions to add
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _dreams_add_permissions($role_name, $permissions) {
    $role = user_role_load_by_name($role_name);
    user_role_grant_permissions($role->rid, $permissions);
}

/**
 * Return the redirect array to the list page.
 * @params type array $query.
 * @return array Redirection array.
 */
function dreams_redirect_to_thank_you($query = array()) {
    // Get thank you page url
    $redirect = variable_get('dreams_redirect');

    drupal_goto($redirect, array('query' => $query));
}

/**
 * Return the redirect array to the list page.
 * @params type array $query.
 * @return array Redirection array.
 */
function dreams_redirect_to_base_assessment_form($query = array()) {
    // Get thank you page url
    $redirect = '/dreams/life-assessment';
//    ddd($redirect, true);
    drupal_goto($redirect, array('query' => $query));
}

/**
 * Return the redirect array to the list page.
 * @params type array $query.
 * @return array Redirection array.
 */
function dreams_redirect_form_to_dashboard($query = array()) {
    // Get thank you page url
    return array('dreams/dashboard', array('query' => $query, 'fragment' => ''),);
}

/**
 * Return the redirect array to the list page.
 * @params type array $query.
 * @return array Redirection array.
 */
function dreams_redirect_form_to_progress($duid, $query = array()) {
    // Get thank you page url
    return array('dreams/' . $duid . '/log/report', array('query' => $query, 'fragment' => ''),);
}

/**
 * Return the redirect array to the list page.
 * @params type array $query.
 * @return array Redirection array.
 */
function dreams_redirect_form_to_last_log($duid, $query = array()) {
    // Get thank you page url
    return array('dreams/' . $duid . '/log', array('query' => $query, 'fragment' => ''),);
}

/**
 * Return the redirect array to the list page.
 * @params type array $query.
 * @return array Redirection array.
 */
function dreams_redirect_form_to_journal($duid, $query = array()) {
    // Get thank you page url
    return array('dreams/journal/' . $duid, array('query' => $query, 'fragment' => ''),);
}

/**
 * Return the redirect array to the list page.
 * @params type array $query.
 * @return array Redirection array.
 */
function dreams_redirect_form_to_goals($duid, $query = array()) {
    // Get thank you page url
    return array('dreams/' . $duid . '/goals', array('query' => $query, 'fragment' => ''),);
}

/**
 * Return the redirect array to the list page.
 * @params type array $query.
 * @return array Redirection array.
 */
function dreams_redirect_form_to_goals_edit($duid, $query = array()) {
    // Get thank you page url
    return array('dreams/' . $duid . '/goals', array('query' => $query, 'fragment' => ''),);
}

/**
 * Return the redirect array to the edit page.
 * @params type array $query.
 * @return array Redirection array.
 */
function dreams_redirect_to_goals_edit($duid, $query = array()) {
    // Get thank you page url
    $redirect = '/dreams/' . $duid . '/goals/edit';
    drupal_goto($redirect, array('query' => $query));
}

/**
 * Return the assessment form main sections.
 * @return array
 */
function dreams_base_sections() {
    $sections = [
        'business' => [
            'label' => t('Business Life'),
            'color' => 'E54D19'
        ],
        'creative' => [
            'label' => t('Creative Life'),
            'color' => 'FAB014'
        ],
        'social' => [
            'label' => t('Social Life'),
            'color' => '86B739'
        ],
        'love' => [
            'label' => t('Love Life'),
            'color' => '076EB8'
        ],
        'purpose' => [
            'label' => t('Life Purpose'),
            'color' => '983395'
        ],
    ];
    foreach ($sections as $key => $value) {

        $sections[$key] = array(
            'label' => variable_get('dreams_' . $key . '_label', $value['label']),
            'color' => variable_get('dreams_' . $key . '_color', $value['color'])
        );
    }

    return $sections;
}

/**
 * Return the assessment form categories.
 * @return array
 */
function dreams_base_categories() {
    $categories = [
        'money' => [
            'label' => t('Money & Finance'),
            'section' => 'business',
            'color' => 'cf2b2b',
            'questions' => [
                t("My earnings are at a level I am 100% happy with."),
                t("I have an emergency fund that gives me financial security. "),
                t("I have no credit card debt / I do not pay interest on any of my debt except my mortgage. "),
                t("I have a budget and stick to it. "),
                t("I save 10% or more of my income every month. "),
                t("My retirement plan will is sufficient to provide the lifestyle I desire. "),
                t("My family is taken care of through insurance should anything happen to me."),
                t("I consciously choose how I spend my money and make wise decisions. "),
                t("I have a diversified investment portfolio. "),
                t("My estate and last will and testament is all in place and up to date. ")
            ]
        ],
        'career' => [
            'label' => t('Career & Mission'),
            'section' => 'business',
            'color' => 'ec6920',
            'questions' => [
                t("My current business/career gives me success and fulfillment. "),
                t("My current business/career has a high probability of me achieving my financial goals for the next 6 months "),
                t("My current business/career path is inline with my long term goals. "),
                t("My work would be fulfilling even if I wasn't been financial compensated."),
                t("I am excited to get out of bed and got to work everyday. "),
                t("I have a clear plan of my professional development to increase my value. "),
                t("My work does not eat into other areas of my life that are important e.g. family, hobbies"),
                t("I always have a clear plan for what I will achieve the next day. "),
                t("I regularly review my written goals and check in on the progress. "),
                t("My work positively impacts others at a depth and scale I am delighted with. ")
            ]
        ],
        'health' => [
            'label' => t('Health & Fitness'),
            'section' => 'creative',
            'color' => 'fdcc04',
            'questions' => [
                t("I know exactly how much water my body needs and I drink enough every day. "),
                t("I am not dependent on caffeine to sustain my energy. "),
                t("I am conscious of and make positive choices about the food I put in my body. "),
                t("I have a healthy breakfast to start each day. "),
                t("My routine includes at least 20 minutes spent outside everyday. "),
                t("I proactively work on my flexibility and posture to keep my body functioning at its peak. "),
                t("I complete 3x30 min cardio workouts every week. "),
                t("I have a fitness routine I love and stick to effortlessly. "),
                t("I have a regular bed time and feel fully rested when I wake up. "),
                t("I never sit for longer than 50 minutes at a time without stretching.")
            ]
        ],
        'fun' => [
            'label' => t('Fun & Recreation'),
            'section' => 'creative',
            'color' => 'f89918',
            'questions' => [
                t("I am always creating new experiences and adventure. "),
                t("I live each day as if it were a gift and precious"),
                t("I am present in every moment throughout the day. "),
                t("I am always making magic moments with my friends and they feel loved and cherished by me."),
                t("I have the perfect amount of time with my friends."),
                t("I have hobbies outside of work that nourish me and I have enough time for them. "),
                t("I have time to breathe and do whatever I want (or nothing at all) every single day. "),
                t("I attend unique events (e.g., opera, museums, theatre) as often as I want to. "),
                t("I have enough time and energy to do what I want and what I need to do everyday. "),
                t("I have a vacation and shut off all communications at least once / year")
            ]
        ],
        'enviroment' => [
            'label' => t('Environment'),
            'section' => 'social',
            'color' => '96c744',
            'questions' => [
                t("My home environment is always clean and tidy. "),
                t("I decline activities and requests that do not fit with my core values. "),
                t("I am always looking for new technologies that can improve my environment."),
                t("My work environment is always tidy and clean."),
                t("I avoid negative influences in my life whether that be people, the media or habits. "),
                t("My life does not contain drama. "),
                t("I have setup my bedroom as a sanctuary for sleep."),
                t("I have healthy food available and limit access to junk food."),
                t("I have messages and images on my phone and wall that empower me."),
                t("I'm surrounded by positive people who raise my standards.")
            ]
        ],
        'community' => [
            'label' => t('Community'),
            'section' => 'social',
            'color' => '7ba73d',
            'questions' => [
                t("I have a community of people who support me."),
                t("My communities support me in many different areas of life."),
                t("I am proactively adding value to my communities. "),
                t("I feel like I am a key part of my communities"),
                t("I am not taking on too much responsibility within all my communities"),
                t("I keep an up to date stakeholder map"),
                t("I am contributing to the growth of my community"),
                t("I am expanding my own sense of community"),
                t("I contribute to communities that I may not be part of but need my help"),
                t("I seek out new communities to be part of.")
            ]
        ],
        'family' => [
            'label' => t('Family & Friends'),
            'section' => 'love',
            'color' => '0079ca',
            'questions' => [
                t("My social circle is a positive influence in my life. "),
                t("I spend time with friends at least every week. "),
                t("I am always thinking about how to increase the success and happiness of my friends and family. "),
                t("I always operate from my highest truth and authentic communication. "),
                t("I seek out help and support when I need it. "),
                t("I have scheduled famly time each week and nothing gets in the way. "),
                t("I honor my commitments I make to others and hold others to the same standard. "),
                t("I trust my colleagues and friends. "),
                t("I do not hold any grudges and have completely forgiven everyone in my life. "),
                t("I have ownership for all conflicts in relationships when they are present. ")
            ]
        ],
        'love' => [
            'label' => t('Partners & Love'),
            'section' => 'love',
            'color' => '0159a9',
            'questions' => [
                t("I am open with and communicate my feelings"),
                t("I appreciate my partner daily and share it with them"),
                t("We schedule a date night and stick to it at least once/week"),
                t("We have an exciting and passionate sex life"),
                t("I encourage my partner to be open and share their feelings with me"),
                t("I am constantly learning how to be a better partner"),
                t("Every 90 days my partner and I take time away just the two of us. "),
                t("We deal with conflicts immediately so things do not build up."),
                t("I show my love through my partners love langauge not my own. "),
                t("I share with my partner how I like to receive love based on my love language")
            ]
        ],
        'learning' => [
            'label' => t('Growth & Learning'),
            'section' => 'purpose',
            'color' => '893994',
            'questions' => [
                t("I acknowledge what I am grateful for every single day without fail. "),
                t("I have a personal growth roadmap and feel on track. "),
                t("I review my habits and goals everyday. "),
                t("I listen to instructional and inspirational content for a minimum of 30 minutes each day. "),
                t("I read instructional or inspirational content for a minimum of 30 minutes each day. "),
                t("I expand my unique gifts and value every day "),
                t("I am up to date with relevant news in my industry"),
                t("I do not engage in gossip and draining chatter"),
                t("I have a mentor who keeps me focused on progress. "),
                t("I have a mentor who I know can help me deliver my desired results faster than I could alone")
            ]
        ],
        'spirituality' => [
            'label' => t('Spirituality'),
            'section' => 'purpose',
            'color' => 'af31a7',
            'questions' => [
                t("I live in alignment with my spiritual beliefs. "),
                t("I feel a deep connection to my spiritual source. "),
                t("I meditate and reflect on my life at least 20 minutes per day. "),
                t("My actions towards others reflect my spiritual beliefs. "),
                t("I use my spirituality to guide me in resolving problems I face. "),
                t("I practice my spirituality daily. "),
                t("I believe myself to be a spiritual person. "),
                t("My spiritality gives me a source of strength and comfort. "),
                t("My spirituality grows the image of my highest self. "),
                t("I use my spirituality to help others. ")
            ]
        ],
    ];


    foreach ($categories as $key => $value) {
        $categories[$key]['label'] = variable_get('dreams_' . $key . '_' . $value['section'] . '_label', $value['label']);
        $categories[$key]['color'] = variable_get('dreams_' . $key . '_' . $value['section'] . '_color', $value['color']);

        foreach ($value['questions'] as $k => $v) {
            $value['questions'][$k] = variable_get('dreams_' . $key . '_' . $value['section'] . '_' . $k, $v);

//            $form['assessment'][$key . '_' . $value['section']]['dreams_' . $key . '_' . $value['section'] . '_' . $k] = array(
//                '#title' => $k + 1 . '. ' . t('question'),
//                '#type' => 'textfield',
//                '#size' => 100,
//                '#default_value' => variable_get('dreams_' . $key . '_' . $value['section'] . '_' . $k, $v),
//                '#required' => TRUE,
//            );
        }
    }

    return $categories;
}

/**
 * Return the assessment form questions.
 * @return json
 */
function dreams_assessments() {
    $questions = [
        ''
    ];
}

/**
 * Get dreams user
 * @param string $duid The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_get_user_by_duid($duid) {
    $result = db_query_range('SELECT * FROM {dreams_user} WHERE duid = :duid', 0, 1, array(':duid' => $duid))->fetchObject();
    return $result;
}

/**
 * Get dreams user
 * @param string $duid The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_get_user_by_duid_assoc($duid) {
    $result = db_query_range('SELECT * FROM {dreams_user} WHERE duid = :duid', 0, 1, array(':duid' => $duid))->fetchAssoc();
    return $result;
}

/**
 * Get dreams user
 * @param string $uid The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_get_user_by_uid($uid) {
    $result = db_query_range('SELECT * FROM {dreams_user} WHERE uid = :uid', 0, 1, array(':uid' => $uid))->fetchObject();
    return $result;
}

/**
 * Get dreams user
 * @param string $uid The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_get_user_by_uid_assoc($uid) {
    $result = db_query_range('SELECT * FROM {dreams_user} WHERE uid = :uid', 0, 1, array(':uid' => $uid))->fetchAssoc();
    return $result;
}

/**
 * Get dreams user's last log
 * @param string $duid The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_get_user_last_log($duid) {
    $result = db_select('dreams_progress_log', 'l')
            ->fields('l')
            ->condition('l.duid', $duid)
            ->orderBy('l.id', 'desc')
            ->execute()
            ->fetchAll();
    return $result[0];
}

/**
 * Get dreams user's all log
 * @param string $duid The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_get_user_logs($duid) {
    $result = db_select('dreams_progress_log', 'l')
            ->fields('l')
            ->condition('l.duid', $duid)
            ->execute()
            ->fetchAll();
    return $result;
}

/**
 * Get dreams user's last habits record
 * @param string $duid The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_get_user_last_habits($duid) {
    $result = db_select('dreams_habits', 'h')
            ->fields('h')
            ->condition('h.duid', $duid)
            ->orderBy('created_at', 'desc')
            ->execute()
            ->fetchAll();
    if (empty($result)) {
        return null;
    }
    return $result[0];
}

/**
 * Get dreams user's journal records
 * @param string $duid The hash code.
 * @return boolean TRUE if the hash code unique.
 * @author Kulcsár Balázs <kulcsar.balazs@gmail.com>
 */
function dreams_get_user_journals($duid) {
    $result = db_select('dreams_journal', 'j')
            ->fields('j')
            ->condition('j.duid', $duid)
            ->orderBy('created_at', 'desc')
            ->execute()
            ->fetchAll();
    return $result;
}
