<?php

/**
 * @file The global functions for the Certify module.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */

/**
 * Check if a value is an integer.
 * @param mixed $variable The value.
 * @return boolean Returns TRUE if the value is an integer else returns FALSE.
 * @author Müller Péter <egyeves@gmail.com>
 */
function _astro_is_int($variable) {
    return ctype_digit(strval($variable));
}

/**
 * Get the fields form the certify_candidate_groups table.
 * @return string The name field value.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_get_user_fields($uid) {
    return db_query_range('SELECT uid, name, mail FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid,))->fetchAssoc();
}

/**
 * Easy look debug function.
 * @params type $var array.
 * @params type $json boolean flag.
 * @return string Formatted array tree.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function add($var, $json = FALSE) {
    if ($json) {
        header('Content-Type: application/json');
        echo drupal_json_output($var);
        exit();
    } else {
        echo '<pre>';
        print_r($var);
        echo '</pre>';
    }
}

/**
 * Filter the array if the value is '' or null or false give back empty string;
 * @param type $form_state It has to contain [values] associative array
 * @return string
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_array_filter_empty_value($value, $form_state) {
    if (is_array($value) && isset($form_state['values']['array_filter'])) {
        $value = array_keys(array_filter($value));
        return $value;
    }
    return $value;
}

/**
 * Price value formatter.
 * @param type number.
 * @return string Formatted.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_format_price($number) {
    return number_format($number, 0, ',', ' ') . ' ' . variable_get('astro_firm_invoice_currency', 'Ft');
}

/**
 * Check if a string is serialized
 * @param string $string
 * @return boolean.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_is_serial($string) {
    return (@unserialize($string) !== false);
}

/**
 * Ellenőrzi az e-mail címet, és hogy az e-mail, domainja létezik-e. 
 * 
 * @param string $email
 * Az email cím (nyers input).
 * 
 * @return boolean
 * TRUE-t ad vissza, ha jó az e-mail formát és azt, hogy a domain létezik.
 * 
 */
function _astro_is_valid_email($email) {
  $isValid = true;
  $atIndex = strrpos($email, "@");
  if (is_bool($atIndex) && !$atIndex) {
    $isValid = false;
  } else {
    $domain = substr($email, $atIndex+1);
    $local = substr($email, 0, $atIndex);
    $localLen = strlen($local);
    $domainLen = strlen($domain);
    if ($localLen < 1 || $localLen > 64){
      // Local part length exceeded.
      $isValid = false;
    } else if ($domainLen < 1 || $domainLen > 255) {
      // Domain part length exceeded.
      $isValid = false;
    } else if ($local[0] == '.' || $local[$localLen-1] == '.') {
      // Local part starts or ends with '.'.
      $isValid = false;
    } else if (preg_match('/\\.\\./', $local)) {
      // Local part has two consecutive dots.
      $isValid = false;
    } else if (!preg_match('/^[A-Za-z0-9\\-\\.]+$/', $domain)) {
      // Character not valid in domain part.
      $isValid = false;
    } else if (preg_match('/\\.\\./', $domain)) {
      // Domain part has two consecutive dots.
      $isValid = false;
    } else if(!preg_match('/^(\\\\.|[A-Za-z0-9!#%&`_=\\/$\'*+?^{}|~.-])+$/', str_replace("\\\\","",$local))) {
      // Character not valid in local part unless local part is quoted.
      if (!preg_match('/^"(\\\\"|[^"])+"$/', str_replace("\\\\","",$local))) {
          $isValid = false;
      }
    }
    if ($isValid && !(checkdnsrr($domain,"MX") || checkdnsrr($domain,"A"))) {
      // Domain not found in DNS.
      $isValid = false;
    }
  }
  return $isValid;
}

/**
 * Check if a string is serialized
 * @param string $bigger
 * @param string $smaller
 * @return absolute value number.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_get_abs_different($bigger, $smaller) {
    return abs(floatval($bigger) - floatval($smaller));
}

/**
 * Check if the uid is valid.
 * @param string $uid The hash code.
 * @return boolean FALSE if the uid is not valid.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_isset_uid($uid) {
    $result = db_query_range('SELECT count(uid) uid FROM {users} WHERE uid = :uid', 0, 1, array(':uid' => $uid))->fetchField();
    return (empty($result)) ? FALSE : TRUE;
}

/**
 * Adds a new role
 * @machine_name - the name of the role to be added
 * @weight - the optional display weight for role
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_add_role($machine_name, $weight = 0) {
    $role = new stdClass();
    $role->name = $machine_name;
    $role->weight = $weight;
    user_role_save($role);
}

/**
 * Adds permissions to a role
 * @role_name - the name of the role to receive an update
 * @permissions - the array of permissions to add
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_add_permissions($role_name, $permissions) {
    $role = user_role_load_by_name($role_name);
    user_role_grant_permissions($role->rid, $permissions);
}

/**
 * Return the two digits string
 * @params type integer.
 * @return string.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_two_digits($int) {
    if (strlen($int) == 1) {
        return '0' . $int;
    } else {
        return $int;
    }
}

/**
 * Add time value to the given date.
 * @params type string date.
 * @params type string.
 * @params type integer.
 * @return timestamp.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_add_time($date, $type = 'days', $int = 1) {

    $new_date = strtotime("+" . $int . " " . $type, strtotime($date));
    return $new_date;
}

/**
 * Show full name in set order.
 * @params type string $lastname.
 * @params type string $firstname.
 * @return string Fullname.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function _astro_show_name($lastname, $firstname) {
    if (variable_get('astro_basic_name', '1') == 1) {
        return _capitalize($firstname) . ' ' . _capitalize($lastname);
    } else {
        return _capitalize($lastname) . ' ' . _capitalize($firstname);
    }
}

/**
 * Egy e-mailt küld a rendszerből a felhasználónak.
 *
 * @param string $name
 * A címzett neve.
 *
 * @param type string $email.
 * A címzett e-mail címe.
 *
 * @param type string $subject.
 * Az e-mail tárgya.
 *
 * @param string $message
 * Az e-mail üzetnet.
 *
 * @author Müller Péter <egyeves@gmail.com>
 */
function _astro_email_send($name, $email, $subject, $message) {

    $settings_mail_from_name = variable_get('certify_email_sender_name');
    $settings_mail_from_email = variable_get('certify_email_sender_email');

    $to = '"' . '=?UTF-8?B?' . base64_encode($name) . '?=' . '" <' . $email . '>';

    $header = 'MIME-Version: 1.0' . PHP_EOL;
    $header .= 'Content-type: text/plain; format=flowed; charset="UTF-8"' . PHP_EOL;
    $header .= 'Content-Transfer-Encoding: 8bit' . PHP_EOL;
    $header .= 'From: "' . '=?UTF-8?B?' . base64_encode($settings_mail_from_name) . '?=' . '" <' . $settings_mail_from_email . '>' . PHP_EOL;
    $header .= 'Reply-To: ' . $settings_mail_from_email . PHP_EOL;
    $header .= 'Return-Path: ' . $settings_mail_from_email . PHP_EOL;
    $header .= 'X-Mailer: PHP-' . phpversion() . PHP_EOL;

    mail($to, '=?UTF-8?B?' . base64_encode($subject) . '?=', $message, $header);
}

/**
 * Get curl function
 * @param string $url
 * @param array $params
 * 
 * @return mixed|json
 */
function _astro_curl_get($url, $params) {
    
    $link = $url . '?' . http_build_query($params);
    
    // Get cURL resource
    $curl = curl_init();
    // Set some options - we are passing in a useragent too here
    curl_setopt_array($curl, [
        CURLOPT_RETURNTRANSFER => 1,
        CURLOPT_URL => $link,
        CURLOPT_USERAGENT => 'Astro cURL Request'
    ]);
    // Send the request & save response to $resp
    $resp = curl_exec($curl);
    // Close request to clear up some resources
    curl_close($curl);
    
    return $resp;
}


/**
 * Show full name in set order.
 * @params type string $lastname.
 * @params type string $firstname.
 * @return string Fullname.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function astro_get_path() {
    return DRUPAL_ROOT . '/' . drupal_get_path('module', 'astro') . '/external/';
}

function astro_get_url() {
    global $base_url;
    return $base_url . '/' . drupal_get_path('module', 'astro') . '/external/';
}

function astro_get_city_suggestions($query, $country_code = 'HUN') {
    global $language;
    
    // http://autocomplete.geocoder.api.here.com/6.2/suggest.json?app_id=z4bFkrOIqnVTzQ5kGi7T&app_code=6Jl9YpjpKHpstrVFW_FR_Q&query=Budapest
    
    // Get here city list
    $default_params = [
      'app_id' => variable_get('astro_here_app_id'),
      'app_code' => variable_get('astro_here_app_code'),
      'country' => $country_code,
      'language' => $language->language,
//      'resultType' => 'postalCode', // areas or postalCode
      'query' => $query,
      'maxresults' => 100
    ];
    
    $url = 'http://autocomplete.geocoder.api.here.com/6.2/suggest.json';
    
    $data = json_decode(_curl_get($url, $default_params));
    
    return $data;
}

/**
 * Beoldvassa a lapozó paramétert az oldalról (page) és visszaadja tisztítás
 * után.
 * 
 * @return string
 * Az olal azonosító sorszáma.
 */
function _astro_get_page() {
  if (isset($_GET['page'])) {
    $page = trim(check_plain($_GET['page']));
  } else {
    $page = '';
  }
  return $page;
}