<?php

/**
 * @file Barion API functions.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */

/**
 * Get vat list.
 * @return type array All vat category.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_get_vat() {
    
    // Set request params
    $params['type']     = 'get';
    $params['function'] = 'vat';
    
    // Get response
    $result = billingo_api_build_request($params);
    
    // Check response
    if ($result->data == ''){
        return FALSE;
    }
    
    // Return response data
    return json_decode($result->data);
    
}

/**
 * Get clients or client.
 * @params type array $id Client id.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_get_client($id = '') {
    
    // Check id
    $id   = check_plain(trim($id));
    
    // Set request params
    $params['type']     = 'get';
    $params['function'] = 'clients';
    if($id != ''){
        $params['id'] = $id;
    }
    
    // Get response
    $result = billingo_api_build_request($params);
    
    // Check response
    if ($result->data == ''){
        return FALSE;
    }
    
    // Return response data
    return json_decode($result->data);
    
}

/**
 * Get products or products.
 * @params type array $id Client id.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_get_product($id = '', $object = FALSE) {
    
    // Check id
    $id   = check_plain(trim($id));
    
    // Set request params
    $params['type']     = 'get';
    $params['function'] = 'products';
    if($id != ''){
        $params['id'] = $id;
    }
    
    // Get response
    $result = billingo_api_build_request($params);
    
    // Check response
    if ($result->data == ''){
        return FALSE;
    }
    
    if($object){
        // Return response data
        return $result;
    } else {
        // Return response data
        return json_decode($result->data);
    }
    
}

/**
 * Get invoices or invoice.
 * @params type array $id Client id.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_get_invoice($id = '') {
    
    // Check id
    $id   = check_plain(trim($id));
    
    // Set request params
    $params['type']     = 'get';
    $params['function'] = 'invoices';
    if($id != ''){
        $params['id'] = $id;
    }
    
    // Get response
    $result = billingo_api_build_request($params);
    
    // Check response
    if ($result->data == ''){
        return FALSE;
    }
    
    // Return response data
    return json_decode($result->data);
    
}

/**
 * Get invoices or invoice.
 * @params type array $id Client id.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_generate_invoice($id = '') {
    
    // Check id
    $id   = check_plain(trim($id));
    
    // Set request params
    $params['type']     = 'get';
    $params['function'] = 'generate';
    if($id != ''){
        $params['id'] = $id;
    } else {
        return FALSE;
    }
    
    // Get response
    $result = billingo_api_build_request($params);
    
    // Check response
    if ($result->data == ''){
        return FALSE;
    }
    
    $invoice = billingo_api_get_invoice_details($id);
    
    $data = json_decode($result->data);
    $data->vat_id = (int)$invoice['vat_id'];
    $data->payment = (int)$invoice['payment'];
    
    // Insert db
    billingo_api_save_invoice_details($invoice['uid'], $data);
    
    // Return response data
    return json_decode($result->data);
    
}

/**
 * Get invoices or invoice.
 * @params type array $id Client id.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_get_invoice_link($id = '') {
    
    // Check id
    $id   = check_plain(trim($id));
    
    // Set request params
    $params['type']     = 'get';
    $params['function'] = 'invoice_link';
    if($id != ''){
        $params['id'] = $id;
    }
    
    // Get response
    $result = billingo_api_build_request($params);
    
    // Check response
    if ($result->data == ''){
        return FALSE;
    }
    
    // Return response data
    return json_decode($result->data);
    
}

/**
 * Create Invoice.
 * @params type array $invoice Invoice details.
 * @params type string $uid User id.
 * @params type string $pid Product id.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_post_invoice($invoice, $uid = '') {
    
    global $user;
    if( $uid == '' ){
        $uid = $user->uid;
    }
    
    // Get client
    $client  = billingo_api_get_client_id($uid);
    
    // Get invoice data
    $fulfillment = date('Y-m-d');
    $days = variable_get('billingo_api_due_date');
    $due_date = date('Y-m-d',strtotime(date('Y-m-d') . "+$days days"));
    
    $data = [
        "fulfillment_date"      => $fulfillment,
        "due_date"              => $due_date,
        "payment_method"        => (int)$invoice['payment'],
        "comment"               => $invoice['comment'],
        "template_lang_code"    => BILLINGO_API_LANG_CODE,
        "electronic_invoice"    => 1,
        "currency"              => variable_get('billingo_api_currency'),
        "client_uid"            => (int)$client,
        "block_uid"             => 0, 
        "type"                  => (int)$invoice['type'], 
        "round_to"              => 0, 
        "items"        => [
            [
                "description"    => $invoice['item_name'],
                "vat_id"         => (int)$invoice['vat_id'],
                "qty"            => (int)$invoice['qty'],
                "net_unit_price" => (int)$invoice['net_unit_price'],
                "unit"           => $invoice['unit'],
                "item_comment"   => $invoice['item_comment']
            ]
        ]
    ];
    
    // Set request params
    $params['type']     = 'post';
    $params['function'] = 'invoices';
    $params['post']     = $data;
    
    // Get response
    $result = billingo_api_build_request($params);
    
    // Check response
    if ($result->data == ''){
        return FALSE;
    }
    
    $data = json_decode($result->data);
    $data->vat_id = (int)$invoice['vat_id'];
    $data->payment = (int)$invoice['payment'];
    
    // Insert db
    billingo_api_save_invoice_details($uid, $data);
    
    // Save to watchdog.
    watchdog('billingo_api', '"%uid" user invoice is created. Data: %data', array('%uid' => $uid, '%data' => $result->data, WATCHDOG_NOTICE));
    
    // Return response data
    return json_decode($result->data);
    
}

/**
 * Save Billingo product to database.
 * @params type string $uid.
 * @params type array $data Invoice all data.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_save_invoice_details($uid, $data) {
    
    if(!empty($data)){
        
        // Get download link
        $url = billingo_api_get_invoice_link($data->id);
        
        $fields = array(
            'invid'         => $data->id,
            'response'      => serialize($data),
            'uid'           => $uid,
            'type'          => $data->attributes->type_string == "proforma" ? 0 : 1,
            'url'           => $url,
            'price'         => $data->attributes->total,
            'due_date'      => $data->attributes->due_date,
            'comment'       => $data->attributes->comment,
            'invoice_no'    => $data->attributes->invoice_no,
            'vat_id'        => $data->vat_id,
            'payment'       => $data->payment,
            'created'       => date('Y-m-d H:i:s')
        );
        $id = db_merge('billingo_api_invoices')
                ->fields($fields)
                ->condition('invid', $fields['invid'])
                ->execute();

        // Save to watchdog.
        watchdog('billingo_api', 'Saved billingo invoice. Data: %data', array('%data' => json_encode($fields), WATCHDOG_NOTICE));
        
        return $id;
        
    } else {
        
        return FALSE;
    
    }
    
}


/**
 * Get Billingo invoice details from database.
 * @params type string $pid.
 * @return type array $product.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_get_invoice_details($id) {
    
    $invoice = db_select('billingo_api_invoices', 'i')
            ->fields('i')
            ->condition('invid', $id)
            ->execute()
            ->fetchAssoc();
    return $invoice;
    
}

/**
 * Create client.
 * @params type array $company Company details.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_post_client($company) {
    
    // Get company
    $client = [
        "name"      => $company['long_name'],
        "email"     => $company['email'],
        "taxcode"   => $company['taxid'],
        "force"     => false,
        "billing_address" => [
            "street_name" => $company['street_name'],
            "street_type" => $company['street_type'],
            "house_nr"    => $company['street_no'],
            "block"       => "", 
            "entrance"    => "", 
            "floor"       => "", 
            "door"        => "",
            "city"        => $company['city'],
            "postcode"    => $company['zip'],
            "district"    => "", 
            "country"     => $company['country']
        ],
        "phone"     => $company['phone'],
        "bank"      => [
            "account_no"  => "", 
            "iban"        => "", 
            "swift"       => ""
        ],
        "fokonyv_szam"    => "", 
        "type"            => "0", 
        "defaults"        => [
            "payment_method"             => "5",
            "electronic_invoice"         => "0",
            "invoice_due_days"           => variable_get('billingo_api_due_date'),
            "invoice_currency"           => variable_get('billingo_api_currency'),
            "invoice_template_lang_code" => BILLINGO_API_LANG_CODE
        ]  
    ];
    
    // Set request params
    $params['type']     = 'post';
    $params['function'] = 'clients';
    $params['post']     = $client;
    
    // Get response
    $result = billingo_api_build_request($params);
    
    // Check response
    if ($result->data == ''){
        return FALSE;
    }
    
    // Insert db
    billingo_api_save_client_id($company['uid'], json_decode($result->data));
    
    // Set watchdog
    watchdog('billingo_api', 'New client generated. UID: %uid, Client ID: %cid', array(
        '%uid' => $company['uid'],
        '%cid' => json_decode($result->data)->id,
    ), WATCHDOG_NOTICE, l(t('Show'), 'user/' . $company['uid']));
    
    // Return response data
    return json_decode($result->data);
    
}

/**
 * Create client.
 * @params type array $company Company details.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_put_client($company) {
    
    // Build company details
    $client = [
        "name"      => $company['long_name'],
        "email"     => $company['email'],
        "taxcode"   => $company['taxid'],
        "force"     => false,
        "billing_address" => [
            "street_name" => $company['street_name'],
            "street_type" => $company['street_type'],
            "house_nr"    => $company['street_no'],
            "block"       => "", 
            "entrance"    => "", 
            "floor"       => "", 
            "door"        => "",
            "city"        => $company['city'],
            "postcode"    => $company['zip'],
            "district"    => "", 
            "country"     => $company['country']
        ],
        "phone"     => $company['phone'],
        "bank"      => [
            "account_no"  => "", 
            "iban"        => "", 
            "swift"       => ""
        ],
        "fokonyv_szam"    => "", 
        "type"            => "0", 
        "defaults"        => [
            "payment_method"             => "5",
            "electronic_invoice"         => "1",
            "invoice_due_days"           => variable_get('billingo_api_due_date'),
            "invoice_currency"           => variable_get('billingo_api_currency'),
            "invoice_template_lang_code" => BILLINGO_API_LANG_CODE
        ]  
    ];
    
    // Set request params
    $params['type']     = 'put';
    $params['function'] = 'clients';
    $params['id']       = $company['id'];
    $params['post']     = $client;
    
    // Get response
    $result = billingo_api_build_request($params);
    
    // Check response
    if ($result->data == ''){
        return FALSE;
    }
    
    // Merge db
    billingo_api_save_client_id($company['uid'], json_decode($result->data));
    
    // Return response data
    return json_decode($result->data);
    
}

/**
 * Get Billingo client id from database.
 * @params type string $uid.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_get_client_id($uid) {
    
    $id = db_select('billingo_api_clients', 'c')
            ->fields('c', array('cid'))
            ->condition('uid', $uid)
            ->execute()
            ->fetchField();
    return (int)$id;
    
}

/**
 * Get Billingo client id from database.
 * @params type string $uid.
 * @params type array $data Client all data.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_save_client_id($uid, $data = array()) {
    
    if(!empty($data)){
        
        $fields = array(
            'uid'       => $uid,
            'cid'       => $data->id,
            'name'      => $data->attributes->name,
            'taxid'     => $data->attributes->taxcode,
            'email'     => $data->attributes->email
        );
        $id = db_merge('billingo_api_clients')
                ->fields($fields)
                ->condition('uid', $uid)
                ->execute();

        // Save to watchdog.
        watchdog('billingo_api', '"%uid" saved as billingo client. Data: %data', array('%uid' => $uid, '%data' => json_encode($fields), WATCHDOG_NOTICE));
        
        return $id;
        
    } else {
        
        return FALSE;
    
    }
    
}

/**
 * Create client.
 * @params type array $company Company details.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_post_product($product) {
    
    // Get company
    $details = [
        "name"          => $product['name'],
        "price"         => $product['price'],
        "unit"          => $product['unit'],
        "currency_code" => variable_get('billingo_api_currency'),
        "vat_id"        => $product['vat_id']
    ];
    
    // Set request params
    $params['type']     = 'post';
    $params['function'] = 'products';
    $params['post']     = $details;
    
    $pid = billingo_api_get_product_id($details['name'], $details['price'], $details['vat_id']);
    
    // Get response
    if(empty($pid)){
        $result = billingo_api_build_request($params);
    } else {
        $result = billingo_api_get_product($pid)[0];
    }
    
    // Check response
    if ( (isset($result->data) && $result->data == '') && !isset($result->attributes) ){
        return FALSE;
    }
    
    // Insert db
    if(!empty($pid)){
        
        // Return response data
        return $result;
        
    } else {
        // Save product id
        billingo_api_save_product_id(json_decode($result->data));
        
        // Return response data
        return json_decode($result->data);
        
    }
        
}

/**
 * Save Billingo product to database.
 * @params type string $uid.
 * @params type array $data Client all data.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_save_product_id($data = array()) {
    
    if(!empty($data)){
        
        $fields = array(
            'pid'        => $data->id,
            'name'       => $data->attributes->name,
            'price'      => $data->attributes->price,
            'vat_id'     => $data->attributes->vat_id
        );
        $id = db_merge('billingo_api_products')
                ->fields($fields)
                ->condition('pid', $fields['pid'])
                ->execute();

        // Save to watchdog.
        watchdog('billingo_api', 'Saved billingo product. Data: %data', array('%data' => json_encode($fields), WATCHDOG_NOTICE));
        
        return $id;
        
    } else {
        
        return FALSE;
    
    }
    
}

/**
 * Get Billingo product id from database.
 * @params type string $name.
 * @params type string $price.
 * @params type string $vat_id.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_get_product_id($name, $price, $vat_id) {
    
    $id = db_select('billingo_api_products', 'p')
            ->fields('p', array('pid'))
            ->condition('name', $name)
            ->condition('price', $price)
            ->condition('vat_id', $vat_id)
            ->execute()
            ->fetchField();
    return $id;
    
}

/**
 * Get Billingo product details from database.
 * @params type string $pid.
 * @return type array $product.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_get_product_details($pid) {
    
    $product = db_select('billingo_api_products', 'p')
            ->fields('p')
            ->condition('pid', $pid)
            ->execute()
            ->fetchAssoc();
    return $product;
    
}

/**
 * Payment details function.
 * @params type array $paymentId All details of the order.
 * @return type array Complete response.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_payment_get_details($paymentId) {
    
    // Update token
    $response = db_select('barion_api_transactions', 't')
            ->fields('t', array('response'))
            ->condition('payment', $paymentId)
            ->execute()
            ->fetchField();
    
    if(!empty($response)){
        return unserialize($response);
    } else {
        return FALSE;
    }
    
}

/** 
 * Payment process page before redirect.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_payment_thankyou_page() {
    global $_REQUEST;
    
    // Get paymentId
    if(isset($_REQUEST['paymentId'])){
        $paymentId  = check_plain($_REQUEST['paymentId']);
    } else {
        $paymentId = '';
    }
    
    // Get response
    $payment_response = barion_api_payment_get_details($paymentId);
    
    // Print a message.
//    drupal_set_message(t('Transaction <em>successed</em>.<br>Payment id: "%pid"', array('%pid' => $paymentId)));

    // Save to watchdog.
//    watchdog('barion_api', '"%uid" order is successed. Payment id: %pid', array('%uid' => $user->uid, '%pid' => $paymentId, WATCHDOG_NOTICE));
    
    // Return html
    return theme('barion_api_thankyou', array(
        'payment' => $payment_response,
    ));
}

/** 
 * Get library connector url.
 * @return type string $url.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_get_request_url() {
    
    global $user;
    global $base_url;
    $path = drupal_get_path('module', 'billingo_api');
    $url  = $base_url . '/' . $path . '/' . 'libraries/billingo/index.php';
    
    return $url;
    
}

/** 
 * Build and run Billingo request.
 * @params type array $params Contain request parameters.
 * @return type array Response.
 * @author Kulcsár Balázs <kulcsar.balazs69@gmail.com>
 */
function billingo_api_build_request($params = array()) {
    
    // Set url details
    $url  = billingo_api_get_request_url();
    
    // Check params
    if(!isset($params['function']) && !isset($params['data']) && !isset($params['type'])){
        return FALSE;
    }
    $function = check_plain($params['function']);
    $type     = check_plain($params['type']);
    $id       = isset($params['id'])   ? check_plain($params['id'])   : '';
    $data     = isset($params['data']) ? check_plain($params['data']) : '';
    $post     = isset($params['post']) ? json_encode($params['post']) : '';
    
    // Get keys
    $private_key = variable_get('billingo_api_private_key');
    $public_key  = variable_get('billingo_api_public_key');
    
    // Run request
    if($data == ''){
        if ($type == 'get'){
            if(!isset($params['id'])){
                $data = "private_key=$private_key&public_key=$public_key&type=$type&function=$function";
            } else {
                $data = "private_key=$private_key&public_key=$public_key&type=$type&function=$function&id=$id";
            }
        }
        if ($type == 'post'){
            if(isset($params['post'])){
                $data = "private_key=$private_key&public_key=$public_key&type=$type&function=$function&post=$post";
            } else {
                return FALSE;
            }
        }
        if ($type == 'put'){
            if(isset($params['post']) && isset($params['id'])){
                $data = "private_key=$private_key&public_key=$public_key&type=$type&function=$function&post=$post&id=$id";
            } else {
                return FALSE;
            }
        }
    }
    $options = array(
      'method' => 'POST',
      'data' => $data,
      'timeout' => 15,
      'headers' => array(
        'Content-Type' => 'application/x-www-form-urlencoded',
        'Authorization' => 'Bearer 1234545332',
      ),
    );
    // Get response
    $result = drupal_http_request($url, $options);
    
    return $result;
    
}